"""
ogconfig is a module to provide parsing of the output of the ogconfig-cli binary
"""
import re

reference_regex = re.compile(r'^\s*\(\s*(.+?)\s*\)\s*->\s*(.+?)\s*$')


def parse_element(print_output, root_path=None, root_value=None):
    """
    convert pre-decoded string output from ogconfig-cli -c 'print <ELEM>' to json-like structure,
    returning the root path and value as a turple
    NOTE: it's not implemented for all types and not all types give useful data
    See ironman/ogconfig-cli/command.c:461 for more info.
    :param root_path:
    :param root_value:
    :param print_output: output of the print command with console colours already striped
    :return: (config path WITHOUT leading ., value matching the path)
    """
    if not isinstance(print_output, str):
        raise TypeError('invalid value: {}'.format(print_output))

    if root_path is not None:
        if not root_path or root_path[0] != '.':
            root_path = '.' + root_path
        if root_value is None:
            root_value = {}

    for line in print_output.splitlines():
        if line[:6] == 'root: ':
            if root_path is not None:
                raise ValueError('duplicate root line: {}'.format(line))
            root_path = ''
            root_value = {}
            continue

        elem_path = None
        elem_type = None
        elem_raw = None
        elem_value = None

        index = line.find(':')
        if index == -1 or index < 3:
            raise ValueError('unexpected line (stage 1): {}'.format(line))
        if index + 1 >= len(line) or line[index + 1] != ' ':
            raise ValueError('unexpected line (stage 2): {}'.format(line))
        elem_raw = line[index + 2:]
        if line[index - 1] != ')':
            raise ValueError('unexpected line (stage 3): {}'.format(line))
        elem_type = ''
        index -= 2
        while index >= 0 and line[index] != '(':
            elem_type = line[index] + elem_type
            index -= 1
        if index < 1 or line[index - 1] != ' ':
            raise ValueError('unexpected line (stage 4): {}'.format(line))
        elem_path = line[:index - 1]
        if not elem_path or elem_path[0] != '.':
            elem_path = '.' + elem_path

        if elem_type == 'map':
            elem_value = {}
        elif elem_type == 'list':
            elem_value = []
        elif elem_type == 'bool':
            if elem_raw == 'true':
                elem_value = True
            elif elem_raw == 'false':
                elem_value = False
            else:
                raise ValueError('unexpected line (bool): {}'.format(line))
        elif elem_type == 'integer':
            elem_value = int(elem_raw)
        elif elem_type == 'float':
            elem_value = float(elem_raw)
        elif elem_type == 'string':
            if len(elem_raw) < 2 or elem_raw[0] != "'" or elem_raw[len(elem_raw) - 1] != "'":
                raise ValueError('unexpected line (string): {}'.format(line))
            elem_value = elem_raw[1:len(elem_raw) - 1]
        elif elem_type == 'blob':
            elem_value = elem_raw
        elif elem_type == 'reference':
            elem_value = reference_regex.search(elem_raw).group(1, 2)
            elem_value = {
                'id': elem_value[0],
                'key': elem_value[1],
            }
        elif elem_type == 'if':
            elem_value = elem_raw
        elif elem_type == 'one_of':
            elem_value = elem_raw
        else:
            raise ValueError('unexpected line (stage 5): {}'.format(line))

        if root_path is None:
            root_path, root_value = elem_path, elem_value
        elif len(elem_path) <= len(root_path) or elem_path[:len(root_path)] != root_path:
            raise ValueError('unexpected line (stage 6): {}'.format(line))
        else:
            set_element(root_value, elem_path[len(root_path):], elem_value)

    if root_path is not None:
        root_path = root_path[1:]

    return root_path, root_value


def walk_element(target, key):
    """
    walks a target up until the very last level and returns that and the remainder of key
    :param target:
    :param key: valid key starting from target WITH a leading . or [
    :return: (last_target, last_key)
    """
    if not key:
        raise ValueError('empty key')
    sub = ''
    if key[0] == '.':
        key = key[1:]
        while key:
            if key[0] == '.' or key[0] == '[':
                break
            sub += key[0]
            key = key[1:]
    elif key[0] == '[':
        key = key[1:]
        while key:
            if key[0] == ']':
                break
            sub += key[0]
            key = key[1:]
        if not key or key[0] != ']':
            raise ValueError('invalid fragment: {}'.format(key))
        key = key[1:]
        sub = int(sub)
        diff = sub - len(target) + 1
        if diff > 0:
            target += [None] * diff
    else:
        raise ValueError('invalid key: {}'.format(key))
    if key:
        return walk_element(target[sub], key)
    return target, sub


def set_element(target, key, value):
    """
    uses the config notation to set a value based on a path note it's parent must exist
    :param target:
    :param key: valid key starting from target WITH a leading . or [
    :param value:
    :return:
    """
    target, key = walk_element(target, key)
    target[key] = value


def get_element(target, key):
    """
    returns a value extracted from the target based on a key
    :param target:
    :param key: valid key starting from target WITH a leading . or [
    :return:
    """
    target, key = walk_element(target, key)
    return target[key]


def get_interface_firewall_zones(root, device):
    """
    given a parsed config root and device (e.g. 'net1') extracts a list of zone elements
    :param root:
    :param device:
    :return:
    """
    result = []
    for zone in root['firewall']['zones']:
        found = False
        for ref in zone['physifs']:
            if get_element(root, '.' + ref['key'])['device'] == device:
                found = True
                break
        if found:
            result.append(zone)
    return result


def get_interface_connections(root, device):
    """
    given a parsed config root and device (e.g. 'net1') extracts a list of connection elements
    :param root:
    :param device:
    :return:
    """
    result = []
    for connection in root['system']['net']['conns']:
        if get_element(root, '.' + connection['physif']['key'])['device'] == device:
            result.append(connection)
    return result


def get_interface(root, device):
    """
    given a parsed config root and device (e.g. 'net1') extracts any matching system.net.physif (None if no match)
    :param root:
    :param device:
    :return:
    """
    for interface in root['system']['net']['physifs']:
        if interface['device'] == device:
            return interface
    return None
