import json
import os
import random
import re
import subprocess

import falcon.status_codes as http_status
import packaging.version
import requests
from . import (
    schema,
)
import grpc
from google.protobuf import (
    json_format as jsonpb,
)
from netops import (
    deployment,
)


lock_dir = "/tmp/deploy.lock"
hosts_path = "/etc/ansible/hosts"
deploy_status_prefix = "/tmp/."
deploy_log_prefix = "/var/log/deploy_"
deploy_log_suffix = ".log"
module_manifest_path = "/modules"

# Cache the 'modules' file from disk, so we don't have to read from
# the file every time, which is a lot
module_manifest_cache = None

# version_regex is used to strip irrelevant bits out of a version string
version_regex = re.compile(r'^[A-Za-z]*(\d+)\.[A-Za-z]*(\d+)\.[A-Za-z]*(\d+).*$')

# nom_state_address is the address to dial for the nom-state daemon's grpc api
nom_state_address = 'unix:///var/run/nom-state.sock'


def proto_to_dict(msg):
    """
    converts a protobuf message to a dictionary, is an alias for the options
    """
    return jsonpb.MessageToDict(
        msg,
        preserving_proto_field_name=True,
        including_default_value_fields=True,
    )


def get_env(key):
    """
    returns an env variable by key raising a ValueError if it wasn't set or is empty
    """
    value = os.environ.get(key)
    if not value:
        raise ValueError('required env variable: {}'.format(key))
    return value


def get_system_version():
    """
    returns the current lh system version
    """
    return get_env("LH_SYSTEM_VERSION")


def get_api_version():
    """
    returns the current lh api version
    """
    return get_env("LH_API_VERSION")


def get_host_address():
    """
    returns the current lh host address
    """
    return get_env("LH_HOST_ADDRESS")


def get_module_manifest():
    """
    returns the module manifest after reading and decoding it
    """
    global module_manifest_cache
    if module_manifest_cache is None:
        with open(module_manifest_path) as module_manifest:
            module_manifest_cache = json.load(module_manifest)
    if not isinstance(module_manifest_cache, dict):
        message = 'unexpected module manifest: {}'.format(module_manifest_cache)
        module_manifest_cache = None
        raise ValueError(message)
    return module_manifest_cache


def parse_version(version):
    """
    parses a version string as a comparable packaging.version.Version
    :param version:
    :return:
    """
    if not isinstance(version, str):
        raise TypeError('invalid version: {}'.format(version))
    match = version_regex.match(version)
    if not match:
        raise ValueError('invalid version: {}'.format(version))
    return packaging.version.parse('{}.{}.{}'.format(match.group(1), match.group(2), match.group(3)))


def version_supported(module_version):
    """
    returns true if the current lh_system_version is greater or equal to the given version
    """
    module_version = parse_version(module_version)
    system_version = parse_version(get_system_version())
    return system_version >= module_version


def generate_hosts_file(lh_addr, om_node_addresses, og_node_addresses, arm_node_addresses):
    """
    return an Ansible hosts file where:
        lh_addr is the address of the central Lighthouse to deploy to
        om_node_addresses is a list of addresses of OM nodes to deploy to
        og_node_addresses is a list of addresses of OG nodes to deploy to
        arm_node_addresses is a list of addresses of ARM nodes to deploy to
    """
    return """[central]
{}
[remote:children]
remote_om
remote_arm
[remote_om]
{}
[remote_arm]
{}
[remote_og]
{}
""".format(lh_addr,
           '\n'.join(om_node_addresses),
           '\n'.join(arm_node_addresses),
           '\n'.join(og_node_addresses))


def bool_int(value):
    """Return 1 if value is truthy otherwise return 0"""
    if value:
        return 1
    return 0


def run_command(args, shell=False, cwd=None, env=None, timeout=None,
                stdout=None, stderr=None):
    """Run subprocess.Popen(ARGS).wait(ARGS), provided for monkey patching"""
    return subprocess.Popen(args, shell=shell, cwd=cwd, env=env, stdout=stdout, stderr=stderr).wait(timeout=timeout)  # pylint: disable=consider-using-with


class NodeNotFoundException(Exception):
    """LH returned a 404 for one or more nodes specified"""


def get_node_addresses(
        node_ids,
        lh_addr=None,
        lh_version=None,
        req_cert=('/root/.netops/lhvpn/lhvpn_server.crt', '/root/.netops/lhvpn/lhvpn_server.key'),
        req_verify=False,
):
    """
    Performs an API request for each of the given node ids, determining their
    address and extracting them into a list within the returned turple based
    on their type.

    :param node_ids:
    :type node_ids: list
    :param lh_addr:
    :param lh_version:
    :param req_cert:
    :param req_verify:

    :return: (om_node_addresses, og_node_addresses)
    :rtype: (list, list)
    """
    om_node_addresses = []
    og_node_addresses = []
    arm_node_addresses = []

    if lh_addr is None:
        lh_addr = get_host_address()
    if lh_version is None:
        lh_version = get_api_version()

    for node_id in node_ids:
        res = requests.get(
            "https://{}/api/{}/nodes/{}".format(lh_addr, lh_version, node_id),
            cert=req_cert,
            verify=req_verify,
        )

        if res.status_code == 404:
            raise NodeNotFoundException("not found status code for node {}".format(node_id))

        if res.status_code != 200:
            raise ValueError("unexpected status code for node {}: {}".format(node_id, res.status_code))

        node_info = res.json()
        if not node_info or "node" not in node_info or "model" not in node_info["node"]:
            raise ValueError("unexpected response for node {}: {}".format(node_id, json.dumps(node_info)))

        if node_info["node"]["model"].startswith("OM") or node_info["node"]["model"].startswith("VOM"):
            om_node_addresses.append(node_info["node"].get("lhvpn_address"))
        elif node_info["node"]["model"].startswith("CM8"):
            arm_node_addresses.append(node_info["node"].get("lhvpn_address"))
        else:
            og_node_addresses.append(node_info["node"].get("lhvpn_address"))

    return om_node_addresses, og_node_addresses, arm_node_addresses


class Modules:
    """
    Modules: the REST handlers for the modules available for deployment.
    """
    schema = schema.AvailableModules

    def on_get(self, request, response):  # pylint: disable=unused-argument
        """
        on_get: retrieve the list of active modules.
        ---
        description: Retrieve information about the latest (supported) modules
        responses:
            200:
                description: OK
                content:
                    application/json:
                        schema: AvailableModules
        """
        response.status = http_status.HTTP_200

        module_manifest = get_module_manifest()

        # List of modules to return from API
        modules_ret = []

        # Pull list of installed images in registry
        resp = requests.get("http://registry:5000/v2/_catalog")

        try:
            local_registry_catalog = resp.json()
        except ValueError:
            # Json wasn't decoded properly
            local_registry_catalog = None

        if local_registry_catalog is None or local_registry_catalog.get("repositories") is None:
            response.media = {"error": "Could not retrieve installed Netops images from registry"}
            return

        # Find the tags for all installed images, we need the information to determine
        # version differences
        local_registry_images = {}
        for repository in local_registry_catalog.get("repositories"):
            # Pull list of installed images in registry
            resp = requests.get("http://registry:5000/v2/{}/tags/list".format(repository))
            try:
                tags = resp.json()
            except ValueError:
                # Json wasn't decoded properly
                response.media = {
                    "error": "Could not retrieve installed Netops images from registry"
                }
                return
            local_registry_images[repository] = tags.get("tags", [])

        # Iterate over modules file, and find modules that are fully installed
        # (would be nice to flag partially installed). We'll also return what
        # containers are a part of the module. Include only active containers
        for module in module_manifest.get("modules", []):
            versions = module.get("versions")
            if versions is None:
                response.media = {"error": "Invalid modules file"}
                return

            # Find the latest available version for the current LH system version
            latest_supported_version = None
            for version in versions:
                if version_supported(version["min_lh_version"]):
                    latest_supported_version = version
                    break

            if not latest_supported_version:
                continue

            if latest_supported_version["images"] is None:
                response.media = {"error": "Invalid modules file"}
                return

            version_available = True

            # Iterate over the required images for this module, and create a list of locally available
            # tags for each image (named containers in the response but probably the wrong name).
            # If we have all the required versions for each image available locally, set this module
            # status to be 'available'.
            version_available_images = []
            for module_image in latest_supported_version["images"]:
                image_name = module_image.get("image")
                if image_name in local_registry_images:
                    local_registry_image_tags = local_registry_images[image_name]
                    version_available_images.append(
                        {"name": image_name, "tags": local_registry_image_tags})

                    if not module_image.get("version") in local_registry_image_tags:
                        version_available = False
                else:
                    version_available = False

            stat = "available" if version_available else "not available"
            if module.get("active"):
                modules_ret.append({
                    "id": module.get("module"),
                    "module": module.get("module"),
                    "name": module.get("name"),
                    "description": module.get("description"),
                    "status": stat,
                    "version": latest_supported_version.get("version"),
                    "license_required": module.get("license_required"),
                    "containers": version_available_images,
                    "images": ["{}:{}".format(x.get("image"), x.get("version")) for x in latest_supported_version.get("images")],
                    "port": module.get("port"),
                    "min_lh_version": latest_supported_version.get("min_lh_version"),
                    "min_ngcs_version": latest_supported_version.get("min_ngcs_version"),
                    "always_activate": module.get("always_activate"),
                    "retroactively_activate": module.get("retroactively_activate"),
                })

        response.media = {"netops-modules": modules_ret}


class Deploy:
    """
    Deploy: the REST handlers for deploying modules via ansible.
    """

    def on_post(self, request, response):
        """
        ---
        description: Start a deployment
        requestBody:
            content:
                application/json:
                    schema: DeployRequest
        responses:
            200:
                description: OK
                content:
                    application/json:
                        schema: DeployResponse
            202:
                description: Accepted (lie, returned when a deploy is in progress)
                content:
                    application/json:
                        schema: Message
            400:
                description: Bad Request
                content:
                    application/json:
                        schema: Error
        """
        module_manifest = get_module_manifest()

        lh_addr = get_host_address()
        lh_version = get_api_version()

        # parse response body
        try:
            body = json.loads(request.bounded_stream.read())
        except json.JSONDecodeError:
            response.status = http_status.HTTP_400
            response.media = {"error": "Failed to parse request data"}
            return

        # extract and validate parameters from body
        target_module = body.get("module")
        target_nodes = body.get("nodes")
        if target_module is None:
            response.status = http_status.HTTP_400
            response.media = {"error": "No module provided"}
            return
        if target_nodes is None:
            response.status = http_status.HTTP_400
            response.media = {"error": "No nodes provided"}
            return

        # Get deploy to central flag (default true if not provided)
        deploy_central = body.get("deploy_central", True)

        # For each node, retrieve its LHVPN information from the LH REST API
        try:
            om_node_addresses, og_node_addresses, arm_node_addresses = get_node_addresses(
                target_nodes,
                lh_addr=lh_addr,
                lh_version=lh_version,
            )
        except NodeNotFoundException:
            response.status = http_status.HTTP_400
            response.media = {"error": "Could not retrieve node information"}
            return
        except ValueError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Could not retrieve node information"}
            return

        # handle unsupported nodes
        # TODO: Update after phase 2 release
        if ((og_node_addresses or arm_node_addresses) and target_module == 'ag') or (arm_node_addresses and target_module == 'sp'):
            response.status = http_status.HTTP_200
            response.media = {"id": schema.deploy_status_unsupported}
            return

        # mutually exclusive locking on the actual deployment process
        try:
            os.makedirs(lock_dir)
        except OSError:
            response.status = http_status.HTTP_202
            response.media = {"message": "Deployment already in progress"}
            return

        success = False
        try:
            # Generate an ID for this deployment for status checking
            deploy_id = "".join([chr(random.randint(ord('a'), ord('z'))) for i in range(16)])

            # Write out hosts file
            try:
                host_file = open(hosts_path, "w")  # pylint: disable=consider-using-with
            except IOError:
                response.status = http_status.HTTP_500
                response.media = {"error": "Could not open ansible hosts file"}
                return

            host_file.write(generate_hosts_file(lh_addr if deploy_central else "", om_node_addresses, og_node_addresses, arm_node_addresses))
            host_file.close()

            # Get latest supported version to deploy
            latest_supported_version = None
            for module in module_manifest.get("modules", []):
                if module.get("module") == target_module:
                    versions = module.get("versions")
                    for version in versions:
                        if version_supported(version["min_lh_version"]):
                            latest_supported_version = version
                            break

                    if not latest_supported_version:
                        continue

                    if latest_supported_version["images"] is None:
                        response.media = {"error": "Invalid modules file"}
                        return

            if not latest_supported_version:
                response.status = http_status.HTTP_400
                response.media = {"error": "No supported version"}
                return

            # Trigger ansible deployment in background
            run_command(
                "/etc/scripts/deploy {} {} {} {} &".format(
                    target_module,
                    latest_supported_version.get("version"),
                    deploy_id,
                    lh_addr
                ),
                shell=True,
                env=dict(
                    os.environ,
                    APP_DEPLOY_CENTRAL=str(bool_int(deploy_central)),
                ),
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL
            )

            success = True

            response.status = http_status.HTTP_200
            response.media = {"id": deploy_id}
        finally:
            if not success:
                os.rmdir(lock_dir)


class StateStatus:
    """
    StateStatus: the REST handlers for reading nom-state deployment status
    """

    def on_get(self, request, response): # pylint: disable=unused-argument
        """
        ---
        description: Get deployment status from nom-state
        responses:
            200:
                description: OK
                content:
                    application/json:
                        schema: StateStatus
        """
        with grpc.insecure_channel(nom_state_address) as channel:
            state = deployment.NetopsStub(channel).State(deployment.StateRequest()).state

        modules = []
        for planned_spec in state.planned_specs:
            if not planned_spec.HasField('netops_module'):
                continue
            module = {
                'planned_spec': planned_spec,
                'running_spec': None,
                'running_as_planned': False,
                'recent_deployments': [],
                'running_deployments': [],
                'pending_deployments': [],
            }
            for running_spec in state.running_specs:
                if planned_spec.key == running_spec.key:
                    module['running_spec'] = running_spec
                    module['running_as_planned'] = running_spec.HasField('netops_module') and running_spec.netops_module == planned_spec.netops_module
                    break
            modules.append(module)

        def deploy_request_module(deploy_request):
            nonlocal modules
            if deploy_request.central:
                for module in modules:
                    plan = module['planned_spec'].netops_module
                    if (plan.module and deploy_request.module == plan.module) or (plan.product != deployment.PRODUCT_UNKNOWN and deploy_request.product == plan.product):
                        return module
            return None

        for deploy in state.recent_deployments:
            module = deploy_request_module(deploy.request)
            if module:
                module['recent_deployments'].append(deploy)

        for deploy_request in state.running_deployments:
            module = deploy_request_module(deploy_request)
            if module:
                module['running_deployments'].append(deploy_request)

        for deploy_request in state.pending_deployments:
            module = deploy_request_module(deploy_request)
            if module:
                module['pending_deployments'].append(deploy_request)

        last_failed_deploy = None
        is_running = False
        is_pending = False
        for module in modules:
            if not module['running_as_planned'] and module['recent_deployments']:
                deploy = module['recent_deployments'][len(module['recent_deployments'])-1]
                if not deploy.success:
                    if not last_failed_deploy or last_failed_deploy.started < deploy.started:
                        last_failed_deploy = deploy
            if module['running_deployments']:
                is_running = True
            if module['pending_deployments']:
                is_pending = True

            module['planned_spec'] = proto_to_dict(module['planned_spec'])
            module['running_spec'] = proto_to_dict(module['running_spec'])
            for i in range(len(module['recent_deployments'])):
                module['recent_deployments'][i] = proto_to_dict(module['recent_deployments'][i])
            for i in range(len(module['running_deployments'])):
                module['running_deployments'][i] = proto_to_dict(module['running_deployments'][i])
            for i in range(len(module['pending_deployments'])):
                module['pending_deployments'][i] = proto_to_dict(module['pending_deployments'][i])

        response.status = http_status.HTTP_200
        if last_failed_deploy:
            response.media = {'status': 'error',
                              'error_message': 'Deployment failed. Refer to syslog for further details.',
                              'modules': modules}
        elif is_running:
            response.media = {'status': 'running', 'error_message': '', 'modules': modules}
        elif is_pending:
            response.media = {'status': 'pending', 'error_message': '', 'modules': modules}
        else:
            response.media = {'status': 'finished', 'error_message': '', 'modules': modules}

class State:
    """
    State: the REST handlers for reading nom-state
    """

    def on_get(self, request, response): # pylint: disable=unused-argument
        """
        ---
        description: Get deployment status from nom-state
        responses:
            200:
                description: OK
                content:
                    application/json:
                        schema: State
        """
        with grpc.insecure_channel(nom_state_address) as channel:
            state = deployment.NetopsStub(channel).State(deployment.StateRequest()).state
        response.status = http_status.HTTP_200
        response.media = proto_to_dict(state)


class Cleanup:
    """
    Cleanup: REST handlers for handling module cleanup
    """

    def on_post(self, request, response):
        """
        ---
        description: Cleanup a deployment
        responses:
            200:
                description: OK
        """
        lh_addr = get_host_address()
        lh_version = get_api_version()

        try:
            body = json.loads(request.bounded_stream.read())
        except json.JSONDecodeError:
            response.status = http_status.HTTP_400
            response.media = {"error": "Failed to parse request data"}
            return

        target_module = body.get("module")
        target_nodes = body.get("nodes")
        if target_module is None:
            response.status = http_status.HTTP_400
            response.media = {"error": "No module provided"}
            return
        if target_nodes is None:
            response.status = http_status.HTTP_400
            response.media = {"error": "No nodes provided"}
            return

        try:
            om_node_addresses, og_node_addresses, arm_node_addresses = get_node_addresses(
                target_nodes,
                lh_addr=lh_addr,
                lh_version=lh_version,
            )
            hosts = om_node_addresses + og_node_addresses + arm_node_addresses
        except NodeNotFoundException:
            response.status = http_status.HTTP_400
            response.media = {"error": "Could not retrieve node information"}
            return
        except ValueError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Could not retrieve node information"}
            return

        error_body = {"errors": {}}
        errors = error_body["errors"]

        cleanup_success = self._cleanup(target_module, hosts)

        if not cleanup_success:
            errors["cleanup"] = "NetOps module '{}' is not supported for cleanup".format(target_module)

        if errors:
            response.status = http_status.HTTP_400
            response.media = error_body
            return

        response.status = http_status.HTTP_200
        return

    @staticmethod
    def _cleanup(target_module, hosts):
        if target_module == 'dop':
            for host in hosts:
                try:
                    subprocess.run(['ssh', '-oBatchMode=yes', f"root@{host}", 'nohup /etc/config/scripts/cleanup_og.sh < /dev/null > /dev/null 2>&1 &'],
                                   shell=False,
                                   check=True,
                                   start_new_session=True,
                                   timeout=10,
                                   stdin=subprocess.DEVNULL,
                                   stdout=subprocess.DEVNULL,
                                   stderr=subprocess.DEVNULL)
                except subprocess.CalledProcessError:
                    # fail silently on non-zero exit code...
                    pass
                except subprocess.TimeoutExpired:
                    # fail silently on command timeout...
                    pass
            return True
        return False


class Status:
    """
    Status: query the deployment status of a running playbook.
    """

    def on_get(self, request, response, deploy_id):  # pylint: disable=unused-argument
        """
        ---
        description: Check status of a deployment
        parameters:
            - name: deploy_id
              in: path
              schema:
                type: string
        responses:
            200:
                description: OK (potentially a lie)
                content:
                    application/json:
                        schema: DeployStatus
        """
        response.status = http_status.HTTP_200

        # if the deploy_id matches a status we just return it directly, this
        # is used to avoid breaking LH behavior but return a specific status
        # identified in the actual deploy POST
        if deploy_id in schema.deploy_statuses:
            response.media = {"status": deploy_id}
            return

        # since the actual creation of the file with the final status is
        # backgrounded we are forced to return in progress if there exists a
        # lock file
        if os.path.isdir(lock_dir):
            response.media = {"status": schema.deploy_status_in_progress}
            return

        # theoretically if we got this far the deploy script should have made
        # a file containing the exit code
        # NOTE: dunno why this wouldn't be represented via a 404
        deploy_status_file = "{}{}".format(deploy_status_prefix, deploy_id)
        # it should have also made a log file but we don't _really_ care about that
        deploy_log_file = "{}{}{}".format(deploy_log_prefix, deploy_id, deploy_log_suffix)

        # NOTE: the behavior below relies on no deploy_id collisions and
        #       doesn't make persisting statuses or logs very easy

        # if there is no status file just bail out with unknown status
        if not os.path.isfile(deploy_status_file):
            response.media = {"status": schema.deploy_status_unknown}
            return

        deploy_status = ""
        deploy_log = "Log file not found"

        # read the status as the last line from the status file (READS ALL)
        with open(deploy_status_file) as status_file:
            for line in status_file:
                deploy_status = line

        # read any log file then remove it
        # NOTE: this means you only get to process any error once it's rather odd
        if os.path.isfile(deploy_log_file):
            with open(deploy_log_file) as log_file:
                deploy_log = log_file.read()
            os.remove(deploy_log_file)

        # we used to have status code handling here but that is no longer
        # necessary, so we simply treat any "non 0 exit code" (or empty file
        # or blank line) as the failure case
        if deploy_status.strip() != "0":
            response.media = {"status": schema.deploy_status_fail, "error": deploy_log}
            return

        # exit code 0 is the only success case aside from deploy_id == "complete"
        response.media = {"status": schema.deploy_status_complete}
