"""
nodes is a module providing some generally useful functions for communicating
with lighthouse.
"""
import json
import re
import requests
from central_sdi import util
from netops.lighthouse import nodes as lh_nodes

requests.urllib3.disable_warnings()


class Error(Exception):
    """
    base class for exceptions
    """

    def __init__(self, msg): # pylint: disable=super-init-not-called
        self._message = msg

    @property
    def message(self):
        """ returns the exception's error message. """
        return self._message

    def __repr__(self):
        return self._message

    def __str__(self):
        return self._message


class Unauthorized(Error):
    """
    raised when the certificate is not authorized to make any request
    """


class Unknown(Error):
    """
    raised when the request failed for an unknown reason, e.g. InternalServerError
    """


class Forbidden(Error):
    """
    raised when the request is not allowed to access the resource
    """


class NotFound(Error):
    """
    raised when the resource was not found on the server
    """


class Invalid(Error):
    """
    raised when the request contained invalid data
    """


def parse_username_string(username_string):
    """
    Parse the specified username string and return the actual username
    plus the node ID.

    Supported username string formats (which match the Lighthouse pmshell formats):
    - username:node_id e.g "john:nodes-1"
    - username:node_name e.g. "john:syd_acm7"
    - username # TODO node pulled dynamically with challenge-response?
    """
    username = None
    node_id = None

    if username_string:
        username = username_string

    username_fields = username_string.split(':', 1)
    if len(username_fields):
        username = username_fields[0]

    if len(username_fields) <= 1:
        return username, node_id

    node = username_fields[1]
    node_id_pattern = re.compile('^nodes-[0-9]+$')
    if node_id_pattern.match(node):
        node_id = node
        return username, node_id

    try:
        node_ids = get_node_ids_for_name(node)
    except (requests.HTTPError, ValueError):
        return username, node_id

    if len(node_ids):
        node_id = node_ids[0]

    return username, node_id


def get_node_ids_for_name(node_name):
    """
    Find the node with the specified name and return its ID.

    Will only return enrolled Opengear nodes.

    Will return multiple IDs if the node name is not unique, it is up to
    the caller to handle this case.
    """
    resp = util.make_request(
        f"https://{util.lh_addr()}/api/{util.get_api_version()}/nodes?ports=false&etags=false&config:status=Enrolled&config:product=opengear&config:name={node_name}")
    resp.raise_for_status()
    data = resp.json()
    return [n['id'] for n in data['nodes'] if n['name'] == node_name]


def get_node_by_id(node_id, token=None):
    """
    Retrieve the specified node from Lighthouse API.

    If a token is provided, it will use that for auth. Else it will use certs.
    """
    resp = util.make_request(f"https://{util.lh_addr()}/api/{util.get_api_version()}/nodes/{node_id}?ports=false&etags=false",
                            token=token)
    resp.raise_for_status()
    return lh_nodes.Node(resp.json()['node'])


def node_authorized(node_id, token):
    """
    Verify whether a user is authorized for IP Access on the specified node.

    It requires a session token to perform the authorization check.

    In the current auth model, Node User rights on a node grant IP Access rights.

    Return True if authorized, as well as the node details.
    """
    try:
        resp = get_node_by_id(node_id, token)
    except requests.exceptions.HTTPError as err:
        # If we get a 200, we are authorized. If 403, we aren't. For all other return codes,
        # raise an error.
        if err.response.status_code == 403:
            return False, None
        raise err

    return True, resp


def find_nodes_details(node_ids=None, token=None):
    """
    Takes a list of node IDs, and returns a list with the details of each node.

    If the list is empty or None, will return all nodes.

    If a token is specified, the returned list will only contain nodes accessible by
    the user of the token.
    """
    url = f"https://{util.lh_addr()}/api/{util.get_api_version()}/nodes?ports=false&etags=false&config:status=Enrolled&product=opengear"

    if node_ids:
        # TODO this could be a huge query and might hit a limit in the browser or API
        # It might be better to retrieve the entire list and filter here
        # TODO an 'in' operator would be great, instead of a bunch of equals
        search_list = []
        for node_id in node_ids:
            search_list.append(
                {"fieldname": "config:_id", "oper": 1, "datatype": 9, "value": node_id, "type": 3})
        search_query = json.dumps({"items": search_list, "type": 2})

        url = f"{url}&json={search_query}"

    resp = util.make_request(url, token)
    resp.raise_for_status()

    return [lh_nodes.Node(n.get('node')) for n in resp.json().get('nodes', [])]
