"""
This module provides REST handlers for IP Access Certificate Management
"""
import re
import falcon.status_codes as http_status
from central_sdi import certs, util
from central_sdi.api import schema


class Cert:
    """
    Handlers for creating and retrieving a certificate.
    """

    def on_get(self, request, response, serial):  # pylint: disable=unused-argument
        """Cert GET endpoint.
        ---
        description: Get a certificate by serial number
        parameters:
            - name: serial
              in: path
              description: Serial number of certificate
              schema:
                type: string
        responses:
            200:
                description: Certificate object
                content:
                    application/json:
                        schema: CertResponse
            404:
                description: Certificate not found
        """
        manager = certs.CertManager(
            cert_cfg=certs.cert_cfg_default(certs.cert_cfg_load()))

        cert = manager.get_cert_by_serial(serial)
        if cert:
            response.media = cert.to_dict()
            response.status = http_status.HTTP_200
            return

        response.media = {"error": "Certificate not found"}
        response.status = http_status.HTTP_404

    def on_delete(self, request, response, serial):  # pylint: disable=unused-argument
        """Cert DELETE endpoint.
        ---
        description: Revoke a certificate by serial number
        parameters:
            - name: serial
              in: path
              description: Serial number of certificate
              schema:
                type: string
        responses:
            204:
                description: Certificate successfully revoked
            404:
                description: Certificate not found
            400:
                description: Certificate already revoked
            500:
                description: Failed to delete certificate
        """
        manager = certs.CertManager(
            cert_cfg=certs.cert_cfg_default(certs.cert_cfg_load()))

        cert = manager.get_cert_by_serial(serial)
        if not cert:
            response.media = {"error": "Certificate not found"}
            response.status = http_status.HTTP_404
            return

        if cert.revoked:
            response.status = http_status.HTTP_400
            response.media = {"error": "Certificate already revoked"}
            return

        if manager.delete_cert(cert):
            response.status = http_status.HTTP_204
            return

        response.media = {"error": "Failed to delete certificate"}
        response.status = http_status.HTTP_500


class Config():
    """
    Handlers for getting and modifying Certificate configuration.
    """
    schema = schema.CertConfig()

    def on_get(self, request, response):  # pylint: disable=unused-argument
        """Cert Config GET endpoint.
        ---
        description: Get the certificate config
        responses:
            200:
                description: Certificate config
                content:
                    application/json:
                        schema: CertConfig
        """
        response.status = http_status.HTTP_200
        response.media = certs.cert_cfg_default(certs.cert_cfg_load())

    def on_put(self, request, response):
        """Cert Config PUT endpoint.
        ---
        description: Update the certificate config
        requestBody:
            content:
                application/json:
                    schema: CertConfig
        responses:
            200:
                description: Updated certificate config
                content:
                    application/json:
                        schema: CertConfig
        """
        certs.cert_cfg_store(request.context.json)
        response.status = http_status.HTTP_200
        response.media = certs.cert_cfg_load()


class Export():
    """
    Handlers for exporting VPN bundles
    """

    def on_get(self, request, response, serial):
        """Cert Export GET endpoint.
        ---
        description: Export a certificate as a VPN config bundle
        parameters:
            - name: serial
              in: path
              description: Serial number of certificate
              schema:
                type: string
        responses:
            200:
                description: VPN config bundle
            400:
                description: Invalid export format
            404:
                description: Certificate not found
            500:
                description: Failed to export certificate bundle
        """
        try:
            bundle_format = certs.ExportFormat(request.params.get("format", "openvpn"))
        except ValueError:
            response.status = http_status.HTTP_400
            response.media = {"error": "Invalid export format"}
            return

        username = request.params.get("username", None)

        manager = certs.CertManager(
            cert_cfg=certs.cert_cfg_default(certs.cert_cfg_load()))

        cert = manager.get_cert_by_serial(serial)
        if not cert:
            response.media = {"error": "Certificate not found"}
            response.status = http_status.HTTP_404
            return

        exporter = certs.Exporter(cert, certs.ca_crt_load(manager), username)
        data = exporter.export(bundle_format)
        zipped_data = util.zip_files(data)

        # Replace invalid filename characters from cert name, and truncate
        safe_name = re.sub('[^0-9a-zA-Z]+', '_', cert.name)[:32]
        if username:
            safe_name = safe_name + '_' + re.sub('[^0-9a-zA-Z]+', '_', username)[:32]

        if data:
            response.content_type = 'application/zip'
            response.downloadable_as = f"{safe_name}.zip"
            response.media = zipped_data
            response.status = http_status.HTTP_200
            return

        response.status = http_status.HTTP_500
        response.media = {"error": "Failed to export certificate bundle"}


class Certs:
    """
    Handlers for creating and retrieving a certificate.
    """
    schema = schema.CertRequest()

    def on_get(self, request, response):  # pylint: disable=unused-argument
        """Certs GET endpoint.
        ---
        description: Get all certificates
        responses:
            200:
                description: List of all certificates
                content:
                    application/json:
                        schema: CertList
        """
        manager = certs.CertManager(
            cert_cfg=certs.cert_cfg_default(certs.cert_cfg_load()))

        response_body = []
        for cert in manager.certs:
            response_body.append(cert.to_dict())

        response.media = {"certs": response_body}
        response.status = http_status.HTTP_200

    def on_post(self, request, response):
        """Certs POST endpoint.
        ---
        description: Create a new certificate
        requestBody:
            content:
                application/json:
                    schema: CertRequest
        responses:
            200:
                description: New certificate
                content:
                    application/json:
                        schema: CertResponse
            400:
                description: Certificate name already exists
            500:
                description: Failed to create certificate
        """
        name = request.context.json.get("name", None)
        common_name = request.context.json.get("common_name", None)

        manager = certs.CertManager(
            cert_cfg=certs.cert_cfg_default(certs.cert_cfg_load()))

        if manager.get_cert_by_name(name) is not None:
            response.status = http_status.HTTP_400
            response.media = {"error": "Certificate name already exists"}
            return

        new_cert = manager.create_cert(name, common_name)
        if not new_cert:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to create certificate"}
            return

        response.media = new_cert.to_dict()
        response.status = http_status.HTTP_200
