"""
errors: contains the deployment api specific errors for Falcon to return.
"""
from falcon import errors


class NotFound(errors.HTTPNotFound):
    """
    NotFound is returned when a resource is not available.
    """
    def __init__(self, error=None, **kwargs):
        super().__init__(**kwargs)
        self.error = error

    @property
    def has_representation(self):
        """
        Override has_representation property to return error details
        """
        return True

    def to_dict(self, obj_type=dict):
        obj = obj_type()
        obj["error"] = self.error
        return obj

def not_found(exception, req, resp, params): # pylint: disable=unused-argument
    """
    not_found: returns an instance of the NotFound handler.
    """
    raise NotFound(error={"type": 4, # Type:communication
                          "level": 1, # Level:Error
                          "text": "Not Found",
                          "code": 28}) # path_not_found


class NotSupported(errors.HTTPMethodNotAllowed):
    """
    NotSupported is returned when a method is requested on a resource that
    is not handled by existing handlers.
    """
    def __init__(self, allowed_methods, error=None, **kwargs):
        super().__init__(allowed_methods=allowed_methods, **kwargs)
        self.error = error

    @property
    def has_representation(self):
        """
        Override has_representation property to return error details
        """
        return True

    def to_dict(self, obj_type=dict):
        obj = obj_type()
        obj["error"] = self.error
        return obj

def not_supported(exception, req, resp, params): # pylint: disable=unused-argument
    """
    not_supported: returns an instance of the NotSupported handler.
    """
    raise NotSupported(allowed_methods=exception.args[0],
                       error={"type": 1, # Type:communication
                              "level": 1, # Level:Error
                              "text": "Method Not Allowed",
                              "code": 39}) # Invalid Request


class Forbidden(errors.HTTPForbidden):
    """
    Forbidden: when a resource is requested and the Authentication token is not
    granted access to that resource.
    """
    def __init__(self, error=None, **kwargs):
        super().__init__(**kwargs)
        self.error = error

    @property
    def has_representation(self):
        """
        Override has_representation property to return error details
        """
        return True

    def to_dict(self, obj_type=dict):
        obj = obj_type()
        obj["error"] = self.error
        return obj
