"""
Dop: the REST API endpoints for the secure provisioning api.
"""
import cgi
import copy
import io
import json
import os
import re
import tempfile
import sshpubkeys
import subprocess
from ipaddress import AddressValueError, IPv4Address, IPv4Network, NetmaskValueError
from pathlib import Path
from filelock import FileLock

import backoff
import falcon.status_codes as http_status
from falcon import util
import requests
import yaml

import central_dop.config
from central_dop.api import authz, logger, schema
from central_dop.nodes import get_api_version, get_host_address
from netops.util import IterableStream

# ui_dir_path is the primary data store used by the api / ui based workflow, and is
# where external resources will be staged (locally) prior to pushing resources
ui_dir_path = '/srv/central-ui/'

yaml_cfg_file = 'config.yml'
sync_file = '/srv/central-ui/.resources_unsynchronised_'
workspace_address = 'http://dop-storage-api'
global_cfg_file = 'doprc'
global_cfg_defaults = {
    'ipaddress': '10.0.0.1',
    'subnet_mask': '255.255.255.0'
}
ssh_keys_file = 'keys'
authorized_keys_file = '/srv/authorized_keys'
authorized_keys_filelock = FileLock(authorized_keys_file + '.lock')  # pylint: disable=abstract-class-instantiated


class ObjectNotFound(Exception):
    """
    indicates that an object was not found at the provided workspace key
    """


# TODO: the workspace "interface" (if you can call it that) below is poorly abstracted and global


def workspace_store(key, value):
    """
    stores arbitrary data in the workspace api, note that value should be an
    iterable for streaming of large files

    :param key:
    :param value:
    """

    requests.post('{}/{}'.format(workspace_address, key), data=value).raise_for_status()


def workspace_load(key):
    """
    loads arbitrary data in the workspace api, streaming the response

    you might want to read this the docs re: streaming behavior
    https://2.python-requests.org/en/master/user/quickstart/#raw-response-content

    :param key:
    :return: stream
    """
    response = requests.get('{}/{}'.format(workspace_address, key), stream=True)
    if response.status_code == 404:
        raise ObjectNotFound('workspace object not found: {}'.format(key))
    response.raise_for_status()
    # TODO: this might be leaky? doc says "shouldn't need to call close directly", dunno
    return io.BufferedReader(IterableStream(response.iter_content()), buffer_size=io.DEFAULT_BUFFER_SIZE)


def workspace_delete(key):
    """
    deletes arbitrary data from the workspace api, succeeding if it didn't exist
    """
    response = requests.delete('{}/{}'.format(workspace_address, key))
    if response.status_code == 404:
        return
    response.raise_for_status()


def workspace_list(key):
    """
    lists all the "files" for a given path, which should be a "directory"
    """
    files = []
    try:
        for item in json.load(workspace_load('{}/index.json'.format(key)))['items']:
            if not item['name'].endswith('/'):
                files.append(item['name'])
    except ObjectNotFound:
        pass
    return files


def global_cfg_default(cfg):
    """
    apply defaults to global config
    :param cfg:
    :return:
    """
    return copy.deepcopy({**global_cfg_defaults, **cfg})


def global_cfg_load():
    """
    fetch global config
    :return:
    """
    try:
        return yaml.safe_load(workspace_load(global_cfg_file))
    except ObjectNotFound:
        return {}


def global_cfg_store(cfg):
    """
    update global config
    :param cfg:
    :return:
    """
    workspace_store(global_cfg_file, yaml.dump({**cfg}))


@backoff.on_exception(backoff.expo, requests.HTTPError, factor=1, max_value=30)
def workspace_store_file(key, path):
    """
    performs a store of a file at a given path, also providing backoff / retry

    :param key:
    :param path:
    :return:
    """
    with open(path, "rb") as value:
        workspace_store(key, value)


def get_user_workspace(token, secctxt):
    """
    gets the name of the workspace for a given user, which is their username unless
    they are a non-root admin user
    """

    # Lighthouse admin users (and root) will share a single resource directory.
    # Can check for LighthouseAdmin role by checking access to settings
    if authz.can(token, '/nom/dop/settings', 'put'):
        return "root"

    return secctxt["username"]


def load_config_file(workspace):
    """
    return the config file for a given workspace
    """
    try:
        return yaml.safe_load(workspace_load(workspace + '/' + yaml_cfg_file))
    except ObjectNotFound:
        return {"device_resources": {}, "node_inventory": {}, "deployment": {}}

def load_keys_file():
    """
    return the ssh keys file
    """
    try:
        return json.load(workspace_load(ssh_keys_file))
    except ObjectNotFound:
        return []

def save_config_file(workspace, config):
    """
    Write the config yaml file to the given workspace
    """
    workspace_store(workspace + '/' + yaml_cfg_file, yaml.dump(config, default_flow_style=False))


def mark_sync(workspace):
    """
    will indicate to a user (or potentially users in the case of admin users) that they need to sync
    """
    Path(sync_file + workspace).touch()


def handle_file_deletion(config, workspace, filename, filetype):
    """ Delete the given file only if it is not used by any remaining resource """
    for _, resource in config.get("device_resources", {}).items():
        if resource.get(filetype) == filename:
            return
    workspace_delete("{}/{}".format(workspace, filename))


def process_request_file(fileData, tmpdir, subdir):
    """ Handles saving the given file from REST API body """

    if not fileData.filename:
        return http_status.HTTP_400, "File name not provided"

    filename = fileData.filename

    # Create subdirectory
    if not os.path.isdir(tmpdir.name + "/" + subdir):
        os.makedirs(tmpdir.name + "/" + subdir)

    with open(tmpdir.name + "/" + subdir + "/" + filename, "wb") as temp_file:
        temp_file.write(fileData.file.read())

    return http_status.HTTP_200, None


def sync_tmpdir(tmpdir, workdir):
    """
    Copy all new config and files from temporary directory to main working directory
    """
    workspace_store_file(workdir + "/" + yaml_cfg_file, tmpdir + "/" + yaml_cfg_file)

    if os.path.isdir(tmpdir + "/downloads"):
        for f in os.listdir(tmpdir + "/downloads"):
            workspace_store_file(workdir + "/downloads/" + f, tmpdir + "/downloads/" + f)

    if os.path.isdir(tmpdir + "/scripts"):
        for f in os.listdir(tmpdir + "/scripts"):
            workspace_store_file(workdir + "/scripts/" + f, tmpdir + "/scripts/" + f)


def update_device_resources(request, response, create, name=None):
    """
    Modify or create the given device resource
    """

    token = request.context.get("token")
    secctxt = request.context.get("secctxt")
    if not secctxt.get("username"):
        response.status = http_status.HTTP_401
        response.media = {"error": "Could not get username"}
        return

    workspace = get_user_workspace(token, secctxt)

    try:
        config = load_config_file(workspace)
    except yaml.YAMLError:
        response.status = http_status.HTTP_500
        response.media = {"error": "Failed to parse YAML config"}
        return

    if not request.content_type or 'multipart/form-data' not in request.content_type:
        response.status = http_status.HTTP_400
        response.media = {"error": "Invalid request content type"}
        return

    # Decode multipart form
    request.env.setdefault('QUERY_STRING', '')
    # Need this for the falcon testing stream to work
    stream = (request.stream.stream if hasattr(request.stream, 'stream') else request.stream)
    form = cgi.FieldStorage(stream, environ=request.env)

    if "body" not in form:
        response.status = http_status.HTTP_400
        response.media = {"error": "Request body not provided"}
        return

    try:
        body = json.loads(form.getvalue("body"))
    except json.JSONDecodeError:
        response.status = http_status.HTTP_400
        response.media = {"error": "Invalid request body"}
        return

    if create:
        name = body.get("name")

    if not name:
        response.status = http_status.HTTP_400
        response.media = {"error": "Name not provided"}
        return

    # If creating new resource, device resource can not already exist
    # If updating, must already exist
    if create and name in config.get("device_resources", {}):
        response.status = http_status.HTTP_400
        response.media = {"error": "Device resource group already exists"}
        return

    if not create and name not in config.get("device_resources", {}):
        response.status = http_status.HTTP_404
        response.media = {"error": "Device resource group not found"}
        return

    # Verify device type
    if not body.get("deviceType"):
        response.status = http_status.HTTP_400
        response.media = {"error": "Device type not provided"}
        return

    if body.get("deviceType") == "cloud_provisioned" and body.get("provisionAfter"):
        response.status = http_status.HTTP_400
        response.media = {"error": "Ordered Provisioning is not supported for cloud provisioned devices"}
        return

    tmpdir = tempfile.TemporaryDirectory()  # pylint: disable=consider-using-with

    if "configFile" in form:
        status, error = process_request_file(form["configFile"], tmpdir, "downloads")
        if error:
            response.status = status
            response.media = {"error": error}
            tmpdir.cleanup()
            return

    if "imageFile" in form:
        status, error = process_request_file(form["imageFile"], tmpdir, "downloads")
        if error:
            response.status = status
            response.media = {"error": error}
            tmpdir.cleanup()
            return

    if "scriptFile" in form:
        status, error = process_request_file(form["scriptFile"], tmpdir, "downloads")
        if error:
            response.status = status
            response.media = {"error": error}
            tmpdir.cleanup()
            return

    if "postProvisionScript" in form:
        status, error = process_request_file(form["postProvisionScript"], tmpdir, "scripts")
        if error:
            response.status = status
            response.media = {"error": error}
            tmpdir.cleanup()
            return

    if "device_resources" not in config:
        config["device_resources"] = {}

    config["device_resources"][name] = {
        "device_type": body.get("deviceType"),
        "mac_address": body.get("macAddresses", []),
        "serial_number": body.get("serialNumbers", []),
        "enrollment": body.get("enrollment") if body.get("deviceType") == "opengear" else None,
        "model": body.get("opengearModel") if body.get("deviceType") == "opengear" else None,
        "provision_after": body.get("provisionAfter", [])
    }

    if "configFile" in form:
        config["device_resources"][name]["config_file"] = form["configFile"].filename
    elif "configFile" in body:
        config["device_resources"][name]["config_file"] = body["configFile"]

    if "imageFile" in form:
        config["device_resources"][name]["image_file"] = form["imageFile"].filename
    elif "imageFile" in body:
        config["device_resources"][name]["image_file"] = body["imageFile"]

    if "scriptFile" in form:
        config["device_resources"][name]["script_file"] = form["scriptFile"].filename
    elif "scriptFile" in body:
        config["device_resources"][name]["script_file"] = body["scriptFile"]

    if "postProvisionScript" in form:
        config["device_resources"][name]["post_provision_script"] = form["postProvisionScript"].filename
    elif "postProvisionScript" in body:
        config["device_resources"][name]["post_provision_script"] = body["postProvisionScript"]

    try:
        with open(tmpdir.name + "/" + yaml_cfg_file, "w") as f:
            yaml.dump(config, stream=f, default_flow_style=False)
        mark_sync(workspace)
    except IOError as err:
        response.status = http_status.HTTP_500
        response.media = {"error": "Failed to save config file: {}".format(str(err))}
        tmpdir.cleanup()
        return

    # Validate the yaml after updating
    try:
        test_config = central_dop.config.load_config_from_dir(tmpdir.name)
    except central_dop.config.Error as err:
        response.status = http_status.HTTP_400
        response.media = {"error": "Failed to load YAML config from file: {}".format(err)}
        tmpdir.cleanup()
        return

    if test_config:
        try:
            central_dop.config.validate_config(test_config)
        except central_dop.config.Error as err:
            response.status = http_status.HTTP_400
            response.media = {"error": str(err)}
            tmpdir.cleanup()
            return
    else:
        response.status = http_status.HTTP_400
        response.media = {"error": "Invalid YAML config"}
        tmpdir.cleanup()
        return

    # Now complete, move temporary files to checkout directory
    sync_tmpdir(tmpdir.name, workspace)

    tmpdir.cleanup()

    # Different status for different endpoints
    response.media = ["OK"]
    return


def update_node_inventory(request, response, create, name=None):
    """
    Modify or create the given node inventory.
    """

    token = request.context.get("token")
    secctxt = request.context.get("secctxt")
    if not secctxt.get("username"):
        response.status = http_status.HTTP_401
        response.media = {"error": "Could not get username"}
        return

    workspace = get_user_workspace(token, secctxt)

    try:
        config = load_config_file(workspace)
    except yaml.YAMLError:
        response.status = http_status.HTTP_500
        response.media = {"error": "Failed to parse YAML config"}
        return

    try:
        body = json.loads(request.bounded_stream.read())
    except json.JSONDecodeError:
        response.status = http_status.HTTP_400
        response.media = {"error": "Request body not provided"}
        return

    if create:
        name = body.get("name")

    if not name:
        response.status = http_status.HTTP_400
        response.media = {"error": "Name not provided"}
        return

    # Validate that name is allowed
    if not re.compile("^[a-zA-Z0-9_]+$").match(name):
        response.status = http_status.HTTP_400
        response.media = {"error": "Invalid name provided, names must be alphanumeric and can only contain an underscore symbol."}
        return

    if "node_inventory" not in config:
        config["node_inventory"] = {}

    if "deployment" not in config:
        config["deployment"] = {}

    node_inventory = config.get("node_inventory")

    # If creating a node inventory, name must not exist
    # If updating, name HAS TO exist
    if create and node_inventory.get(name):
        response.status = http_status.HTTP_400
        response.media = {"error": "Node inventory already exists"}
        return

    if not create and not node_inventory.get(name):
        response.status = http_status.HTTP_400
        response.media = {"error": "Node inventory doesn't exist"}
        return

    headers = {"Authorization": "Token {}".format(token)}
    lh_addr = get_host_address()
    lh_version = get_api_version()

    config["node_inventory"][name] = {}

    if "static" in body:
        # Make sure all hosts in static list are authorized for this user.
        for node in body["static"]:

            resp = requests.get("https://{}/api/{}/nodes/{}".format(lh_addr, lh_version, node),
                                headers=headers, verify=False)

            if resp.status_code == 403:
                response.status = http_status.HTTP_403
                response.media = {"error": "Permission denied"}
                return

            if resp.status_code not in (200, 201):
                response.status = http_status.HTTP_500
                response.media = {"error": "Could not retrieve node information"}
                return

        config["node_inventory"][name]["static"] = body.get("static")

    if "smartgroup" in body:
        # Make sure we are allowed to deploy to the specified smartgroup.
        # Smartgroups are specified by name in the request, but we need its ID so get the list
        # from LH API and find by name

        smartgroup_id = None
        if not authz.can(token, "/nom/dop/node_inventory_dynamic", "post"):
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
            return

        resp = requests.get("https://{}/api/{}/nodes/smartgroups".format(lh_addr, lh_version),
                            headers=headers, verify=False)

        if resp.status_code == 403:
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
            return

        if resp.status_code not in (200, 201):
            response.status = http_status.HTTP_500
            response.media = {"error": "Could not retrieve smartgroup information"}
            return

        try:
            smartgroups = resp.json()
        except ValueError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Could not retrieve smartgroup information"}
            return

        for smartgroup in smartgroups.get("smartgroups", []):
            group_id = smartgroup.get("id")

            if not group_id:
                response.status = http_status.HTTP_500
                response.media = {"error": "Could not get smartgroup ID"}
                return

            # Stop when you find the correct smartgroup
            if smartgroup.get("name") == body.get("smartgroup"):
                smartgroup_id = group_id
                break

        if not smartgroup_id:
            response.status = http_status.HTTP_404
            response.media = {"error": "Smartgroup not found"}
            return

        config["node_inventory"][name]["smartgroup"] = body.get("smartgroup")

    if "distribution" not in body:
        response.status = http_status.HTTP_400
        response.media = {"error": "Distribution not provided"}
        return
    distribution = body["distribution"]
    if not isinstance(distribution, list):
        response.status = http_status.HTTP_400
        response.media = {"error": "Invalid distribution"}
        return

    # Validate all device resource bundles in this distribution exist
    for device_resources in distribution:
        if device_resources not in config.get("device_resources", []):
            response.status = http_status.HTTP_400
            response.media = {"error": "Device resource bundle '{}' not found".format(device_resources)}
            return

    config["deployment"][name] = distribution

    tmpdir = tempfile.TemporaryDirectory()  # pylint: disable=consider-using-with

    try:
        with open(tmpdir.name + "/" + yaml_cfg_file, "w") as f:
            yaml.dump(config, stream=f, default_flow_style=False)
        mark_sync(workspace)
    except IOError as err:
        response.status = http_status.HTTP_500
        response.media = {"error": "Failed to save config file: {}".format(str(err))}
        tmpdir.cleanup()
        return

    # Validate the yaml after updating
    try:
        test_config = central_dop.config.load_config_from_dir(tmpdir.name)
    except central_dop.config.Error:
        response.status = http_status.HTTP_400
        response.media = {"error": "Failed to load YAML config from file {}".format(err)}
        tmpdir.cleanup()
        return

    if test_config:
        try:
            central_dop.config.validate_config(test_config)
        except central_dop.config.Error as err:
            response.status = http_status.HTTP_400
            response.media = {"error": str(err)}
            tmpdir.cleanup()
            return
    else:
        response.status = http_status.HTTP_400
        response.media = {"error": "Invalid YAML config"}
        tmpdir.cleanup()
        return

    sync_tmpdir(tmpdir.name, workspace)
    tmpdir.cleanup()

    # Different status for different endpoints
    response.media = ["OK"]
    return


def validate_config(ipaddress, subnet_mask):
    errmsg = ''
    if not (isinstance(ipaddress, str) and isinstance(subnet_mask, str)):
        errmsg = "Expected type is string with 4 octects"
        return errmsg
    try:
        IPv4Address(ipaddress)  # verify address is valid
        IPv4Network("0.0.0.0/{}".format(subnet_mask))  # verify netmask is valid
    except (AddressValueError, NetmaskValueError) as err:
        print(err)
        errmsg = err
    return errmsg


def run_command(args, shell=False, cwd=None, env=None, timeout=None, stdout=None, stderr=None):
    """Run subprocess.Popen(ARGS).wait(ARGS), provided for monkey patching"""
    return subprocess.Popen(args, shell=shell, cwd=cwd, env=env, stdout=stdout, stderr=stderr).wait(timeout=timeout)  # pylint: disable=consider-using-with


def resource_deploy(token, workspace=None):
    """
    performs a deploy of all (synced, git, etc) resources, which includes a
    workspace sync, if the workspace was provided and it has been initialised
    """
    if not token:
        raise ValueError(f'invalid token: {token}')
    token = f'{token}'
    if not workspace or not workspace_list(workspace):
        workspace = ''
    else:
        workspace = f'{workspace}'
    if run_command(['bash', '-c', 'nohup "$@" </dev/null >/dev/null 2>&1 &', '-', 'bash', '-c', 'USER_TOKEN="$1" USER_WORKSPACE="$2" exec deploy.sh', '-', token, workspace]) != 0:
        raise RuntimeError("failed to background deployment")


class Resources:
    """ API handler for /nom/dop/device_resources endpoints """
    def on_get(self, request, response):
        """Device Resources GET endpoint.
        ---
        description: Get a list of all device resource bundles
        responses:
            200:
                description: List of all device resource bundles
                content:
                    application/json:
                        schema: DeviceResourceList
            401:
                description: Could not get username
            500:
                description: Failed to parse YAML config
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)

        try:
            config = load_config_file(workspace)
        except yaml.YAMLError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        device_resources = config.get("device_resources", {})

        ret = []
        for name, resource in device_resources.items():
            resource["name"] = name
            resource["distribution"] = []
            # fill in the distribution list, where the name of the device resource
            # matches the name in the deployment list
            for distribution, resources in config.get("deployment").items():
                if name in resources:
                    resource["distribution"].append(distribution)
            ret.append(resource)

        response.status = http_status.HTTP_200
        response.media = {"device_resources": ret}

    def on_post(self, request, response):
        """Device Resources POST endpoint.
        ---
        description: Create a new device resource bundle
        responses:
            201:
                description: Successfully created bundle
            400:
                description: Invalid request
            401:
                description: Could not get username
            500:
                description: Failed to parse YAML config
        """
        response.status = http_status.HTTP_201
        update_device_resources(request, response, create=True)


class Resource:
    """ API handler for /nom/dop/device_resources/{name} endpoints """
    def on_get(self, request, response, name):
        """Device Resources GET endpoint.
        ---
        description: Get a device resource bundle by name
        parameters:
            - name: name
              in: path
              description: Name of resource bundle
              schema:
                type: string
        responses:
            200:
                description: Device resource bundle
                content:
                    application/json:
                        schema: DeviceResource
            401:
                description: Could not get username
            404:
                description: Bundle not found
            500:
                description: Failed to parse YAML config
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)

        try:
            config = load_config_file(workspace)
        except yaml.YAMLError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        if not config.get("device_resources") or not config.get("device_resources").get(name):
            response.status = http_status.HTTP_404
            response.media = {"error": "Device resource group not found"}
            return

        resource = config.get("device_resources").get(name)
        resource["name"] = name
        resource["distribution"] = []

        # Get every distribution that has this resource
        if config.get("deployment"):
            for distribution, resources in config.get("deployment").items():
                if name in resources:
                    resource["distribution"].append(distribution)

        response.status = http_status.HTTP_200
        response.media = resource

    def on_put(self, request, response, name):
        """Device Resources PUT endpoint.
        ---
        description: Update a device resource bundle
        parameters:
            - name: name
              in: path
              description: Name of resource bundle
              schema:
                type: string
        responses:
            200:
                description: Successfully updated bundle
            401:
                description: Could not get username
            404:
                description: Bundle not found
            500:
                description: Failed to parse YAML config
        """
        response.status = http_status.HTTP_200
        update_device_resources(request, response, create=False, name=name)

    def on_delete(self, request, response, name):
        """Device Resources DELETE endpoint.
        ---
        description: Delete a device resource bundle
        parameters:
            - name: name
              in: path
              description: Name of resource bundle
              schema:
                type: string
        responses:
            204:
                description: Successfully deleted bundle
            401:
                description: Could not get username
            404:
                description: Bundle not found
            500:
                description: Failed to parse YAML config
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)
        downloads_dir = workspace + '/downloads'
        scripts_dir = workspace + '/scripts'

        try:
            config = load_config_file(workspace)
        except yaml.YAMLError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        if not config.get("device_resources") or not config.get("device_resources").get(name):
            response.status = http_status.HTTP_404
            response.media = {"error": "Device resource group not found"}
            return

        for _, distribution in config.get("deployment", {}).items():
            # Handle situation that name is in distribution multiple times,
            # which shouldn't happen if created using web ui.
            while name in distribution:
                distribution.remove(name)

        deleted_resource = config.get("device_resources").get(name)
        del config.get("device_resources")[name]

        for filetype in ["config_file", "image_file", "script_file"]:
            if deleted_resource.get(filetype):
                handle_file_deletion(config, downloads_dir, deleted_resource.get(filetype), filetype)

        for filetype in ["post_provision_script"]:
            if deleted_resource.get(filetype):
                handle_file_deletion(config, scripts_dir, deleted_resource.get(filetype), filetype)

        # Delete references to this device resource from the provisionAfter of other resources
        for _, resource in config.get("device_resources").items():
            while name in resource.get("provision_after", []):
                resource.get("provision_after").remove(name)

        try:
            save_config_file(workspace, config)
            mark_sync(workspace)
        except IOError as e:
            response.status = http_status.HTTP_500
            response.media = {"error": str(e)}
            return

        response.status = http_status.HTTP_204
        response.content_type = None


class Files:
    """ API handler for /nom/dop/device_resources_files endpoints """
    def on_get(self, request, response):
        """Device Resource files GET endpoint.
        ---
        description: Get all device resource files
        responses:
            200:
                description: Device resource file list
                content:
                    application/json:
                        schema: DeviceResourceFiles
            401:
                description: Could not get username
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)
        downloads = workspace_list(workspace + '/downloads')
        scripts = workspace_list(workspace + '/scripts')

        response.status = http_status.HTTP_200
        response.media = {
            "downloads": downloads,
            "scripts": scripts,
        }
        return


class Routes:
    """
    API handler for /nom/dop/routes endpoints.
    Used to generate UI sidebar links.
    """
    def on_get(self, request, response):
        """Routes GET endpoint.
        ---
        description: Get list of UI routes
        responses:
            200:
                description: UI routes
        """
        token = request.context.get("token")
        response.status = http_status.HTTP_200
        routes = {"Secure Provisioning":[]}
        if authz.can(token, "/nom/dop/device_resources", "get"):
            routes["Secure Provisioning"].append({
                "name": "Device Resources",
                "route": "resources"})
        if authz.can(token, "/nom/dop/node_inventory", "get"):
            routes["Secure Provisioning"].append({
                "name": "Resource Distribution",
                "route": "distribution"})

        response.media = routes


class Rights:
    """ API handler for /nom/dop/rights endpoints """
    def on_get(self, request, response):
        """Rights GET endpoint.
        ---
        description: Get user's rights
        responses:
            200:
                description: User rights
        """
        token = request.context.get("token")
        response.status = http_status.HTTP_200

        rights = {
            "resources": {
                "view": authz.can(token, "/nom/dop/device_resources", "get"),
                "create": authz.can(token, "/nom/dop/device_resources", "post"),
                "edit": authz.can(token, "/nom/dop/device_resources", "put"),
                "delete": authz.can(token, "/nom/dop/device_resources", "delete")
            },
            "distribution": {
                "view": True,
                "static_inventory": {
                    "view": authz.can(token, "/nom/dop/node_inventory/static", "get"),
                    "create": authz.can(token, "/nom/dop/node_inventory/static", "post"),
                    "edit": authz.can(token, "/nom/dop/node_inventory/static", "put"),
                    "delete": authz.can(token, "/nom/dop/node_inventory/static", "delete")
                },
                "dynamic_inventory": {
                    "view": authz.can(token, "/nom/dop/node_inventory/dynamic", "get"),
                    "create": authz.can(token, "/nom/dop/node_inventory/dynamic", "post"),
                    "edit": authz.can(token, "/nom/dop/node_inventory/dynamic", "put"),
                    "delete": authz.can(token, "/nom/dop/node_inventory/dynamic", "delete")
                }
            }
        }

        response.media = {"rights": rights}


class Push:
    """ API handler for /nom/dop/push endpoints """
    def on_get(self, request, response):  # pylint: disable=unused-argument
        """Push GET endpoint.
        ---
        description: Get the current resource push state
        responses:
            200:
                description: Current push state
                content:
                    application/json:
                        schema: Push
        """
        response.status = http_status.HTTP_200

        # lock dir is created if script is running
        if os.path.isdir("/tmp/post_receive.lock"):
            response.media = {"status": "in_progress"}
            return

        # read result of last resource push from log file
        if not os.path.isfile("/var/log/deploy.log"):
            # no log file, so who knows
            response.media = {"status": "unknown"}
            return

        # Result of most recent push is on last line
        last_line = ""
        with open("/var/log/deploy.log", "r") as log_file:
            for line in log_file:
                last_line = line

        if "Complete!" in last_line:
            response.media = {"status": "complete"}
        elif "Deploy failed" in last_line:
            response.media = {"status": "fail", "error": last_line}
        else:
            response.media = {"status": "unknown"}
        return

    def on_post(self, request, response):  # pylint: disable=unused-argument
        """Push POST endpoint.
        Runs the git post_receive.sh script, to push all resources to nodes.
        ---
        description: Push all resources to nodes
        responses:
            204:
                description: Push started
            400:
                description: Push already in progress
        """
        # lock dir created if script is running
        if os.path.isdir("/tmp/post_receive.lock"):
            response.status = http_status.HTTP_400
            response.media = {"error": "Resource push in progress"}
            return

        token = request.context.get("token")
        # not syncing changes, just redeploying resources
        resource_deploy(token)
        response.status = http_status.HTTP_204
        response.content_type = None


class Sync:
    """ API handler for /nom/dop/sync endpoints """
    def on_get(self, request, response):
        """Sync GET endpoint.
        ---
        description: Get the current resource sync state
        responses:
            200:
                description: Current sync state
                content:
                    application/json:
                        schema: Sync
            401:
                description: Could not get username
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)

        # A sync file is created when the user hasn't deployed files to nodes
        # yet.
        status = "synchronised"
        if os.path.exists(sync_file + workspace):
            status = "unsynchronised"

        response.status = http_status.HTTP_200
        response.media = {"status": status}
        return

    def on_post(self, request, response):
        """Sync POST endpoint.
        ---
        description: Synchronise files between temporary workdir and main UI resource directory.
            Also triggers a resource deploy.
        responses:
            204:
                description: Sync started
            400:
                description: Push already in progress
            401:
                description: Could not get username
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)

        # lock dir created if script is running
        if os.path.isdir("/tmp/post_receive.lock"):
            response.status = http_status.HTTP_400
            response.media = {"error": "Resource push in progress"}
            return

        # Firstly sync the directories and then push the updated resources to
        # nodes.
        resource_deploy(token, workspace)
        response.status = http_status.HTTP_204
        response.content_type = None


class Inventory:
    """ API handler for /nom/dop/node_inventory endpoints """
    schema = schema.NodeInventory

    def on_get(self, request, response):
        """Node Inventory GET endpoint.
        ---
        description: Retrieve the list of node inventories. Inventories will only be
            returned if the user has permission to access information for that node.
        responses:
            200:
                description: List of node inventories
                content:
                    application/json:
                        schema: NodeInventoryList
            401:
                description: Could not get username
            500:
                description: Failed to parse YAML config
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)

        log = logger.set_up_logging()

        try:
            config = load_config_file(workspace)
        except yaml.YAMLError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        all_node_inventory = config.get("node_inventory")
        if all_node_inventory is None:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        distribution = config.get("deployment", {})

        lh_addr = get_host_address()
        lh_version = get_api_version()

        ret = []
        for name, inventory in all_node_inventory.items():
            permitted = True
            inventory["distribution"] = distribution[name] if name in distribution else []

            if inventory.get("static"):
                node_inventory_static = []

                headers = {"Authorization": "Token {}".format(token)}

                for node in inventory.get("static"):
                    # Get list of nodes from LH API
                    resp = requests.get("https://{}/api/{}/nodes/{}".format(lh_addr, lh_version, node),
                                        headers=headers, verify=False)

                    # If node no longer exists, we should skip it. We know it exists
                    # if the response is a 200.
                    if resp.status_code == 200:
                        try:
                            node_info = resp.json()
                        except ValueError:
                            response.status = http_status.HTTP_500
                            response.media = {"error": "Could not retrieve node information"}
                            return
                        node_inventory_static.append("{} ({})".format(node_info["node"]["name"], node))
                    else:
                        log.info(
                            "Could not retrieve node '{}' for node inventory '{}'".format(node, name))

                inventory["static"] = node_inventory_static

            if permitted:
                inventory["name"] = name
                ret.append(inventory)

        response.status = http_status.HTTP_200
        response.media = {"node_inventories": ret}

    def on_post(self, request, response):
        """Node Inventory POST endpoint.
        ---
        description: Create a new node inventory
        responses:
            201:
                description: Successfully created
            400:
                description: Invalid request
            401:
                description: Could not get username
        """
        response.status = http_status.HTTP_201
        update_node_inventory(request, response, True)


class InventoryItem:
    """ API handler for /nom/dop/node_inventory/{name} endpoints """

    def on_get(self, request, response, name):
        """Node Inventory GET endpoint.
        ---
        description: Retrieve a node inventory by name.
        parameters:
            - name: name
              in: path
              description: Name of node inventory
              schema:
                type: string
        responses:
            200:
                description: Node Inventory
                content:
                    application/json:
                        schema: NodeInventory
            401:
                description: Could not get username
            404:
                description: Inventory not found
            500:
                description: Failed to parse YAML config
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)
        log = logger.set_up_logging()

        try:
            config = load_config_file(workspace)
        except yaml.YAMLError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        all_node_inventory = config.get("node_inventory")
        if all_node_inventory is None:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        node_inventory = all_node_inventory.get(name)
        if node_inventory is None:
            response.status = http_status.HTTP_404
            response.media = {"error": "Node inventory not found"}
            return

        # For static hosts, get the node name to display in the UI.
        if node_inventory.get("static"):
            node_inventory_static = []

            headers = {"Authorization": "Token {}".format(token)}
            lh_addr = get_host_address()
            lh_version = get_api_version()

            for node in node_inventory.get("static"):
                # Get list of nodes from LH API
                resp = requests.get("https://{}/api/{}/nodes/{}".format(lh_addr, lh_version, node),
                                    headers=headers, verify=False)

                # If node no longer exists, we should skip it. We know it exists
                # if the response is a 200.
                if resp.status_code == 200:
                    try:
                        node_info = resp.json()
                    except ValueError:
                        response.status = http_status.HTTP_500
                        response.media = {
                            "error": "Could not retrieve node information"}
                        return
                    node_inventory_static.append(
                        "{} ({})".format(node_info["node"]["name"], node))
                else:
                    log.info("Could not retrieve node '{}' for node inventory '{}'".format(node, name))

            node_inventory["static"] = node_inventory_static

        distribution = config.get("deployment", {})
        node_inventory["name"] = name
        node_inventory["distribution"] = distribution[name] if name in distribution else []

        response.status = http_status.HTTP_200
        response.media = node_inventory

    def on_put(self, request, response, name):
        """Node Inventory PUT endpoint.
        ---
        description: Update a node inventory
        parameters:
            - name: name
              in: path
              description: Name of node inventory
              schema:
                type: string
        responses:
            200:
                description: Successfully updated
            400:
                description: Invalid request
            401:
                description: Could not get username
        """
        response.status = http_status.HTTP_200
        update_node_inventory(request, response, False, name)

    def on_delete(self, request, response, name):
        """Node Inventory DELETE endpoint.
        ---
        description: Delete a node inventory
        parameters:
            - name: name
              in: path
              description: Name of node inventory
              schema:
                type: string
        responses:
            204:
                description: Successfully deleted
            400:
                description: Invalid request
            401:
                description: Could not get username
            500:
                description: Failed to parse YAML config
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")
        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        workspace = get_user_workspace(token, secctxt)

        try:
            config = load_config_file(workspace)
        except yaml.YAMLError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Failed to parse YAML config"}
            return

        node_inventory = config.get("node_inventory", {})
        if node_inventory.get(name) is None:
            response.status = http_status.HTTP_404
            response.media = {"error": "Node inventory not found"}
            return

        del config.get("node_inventory")[name]
        deployment = config.get("deployment", {})
        if deployment.get(name):
            del deployment[name]

        try:
            save_config_file(workspace, config)
            mark_sync(workspace)
        except IOError as err:
            response.status = http_status.HTTP_500
            response.media = {"error": str(err)}
            return

        response.status = http_status.HTTP_204
        response.content_type = None


class Smartgroups:
    """ API handler for /nom/dop/smartgroups endpoints """

    def on_get(self, request, response):
        """Smartgroups GET endpoint.
        ---
        description: Retrieve all smartgroups to display on UI. Only smartgroups that
            the user has access to will be returned.
        responses:
            200:
                description: List of smartgroups
                content:
                    application/json:
                        schema: SmartgroupList
            401:
                description: Could not get username
            500:
                description: Could not retrieve smartgroup information
        """
        token = request.context.get("token")
        headers = {"Authorization": "Token {}".format(token)}
        lh_addr = get_host_address()
        lh_version = get_api_version()

        # Get list of smartgroups from LH API
        resp = requests.get("https://{}/api/{}/nodes/smartgroups".format(lh_addr, lh_version),
                            headers=headers, verify=False)

        if resp.status_code != 200:
            response.status = util.misc.get_http_status(resp.status_code)
            response.media = {"error": "Could not retrieve smartgroup information"}
            return

        try:
            smartgroups = resp.json()
        except ValueError:
            response.status = http_status.HTTP_500
            response.media = {"error": "Could not retrieve smartgroup information"}
            return

        ret = []
        for smartgroup in smartgroups.get("smartgroups", []):
            smartgroup_id = smartgroup.get("id")
            if not smartgroup_id:
                response.status = http_status.HTTP_500
                response.media = {"error": "Could not get smartgroup ID"}
                return
            ret.append(smartgroup)

        response.status = http_status.HTTP_200
        response.media = {"smartgroups": ret}


class Config:
    """ API handler for /nom/dop/config for global settings endpoints """
    schema = schema.Config()

    def on_get(self, request, response):
        """Config GET endpoint.
        ---
        description: Retrieve current configuration.
        responses:
            200:
                description: Configuration
                content:
                    application/json:
                        schema: Config
            401:
                description: Could not get username
        """
        secctxt = request.context.get("secctxt")

        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        response.status = http_status.HTTP_200
        response.media = global_cfg_default(global_cfg_load())

    def on_put(self, request, response):
        """Config PUT endpoint.
        ---
        description: Update configuration
        responses:
            204:
                description: Successfully updated
            400:
                description: Invalid request
            401:
                description: Could not get username
        """
        token = request.context.get("token")
        secctxt = request.context.get("secctxt")

        if not authz.can(token, "/nom/dop/config", "put"):
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
            return

        if not secctxt.get("username"):
            response.status = http_status.HTTP_401
            response.media = {"error": "Could not get username"}
            return

        post_data = request.bounded_stream.read()
        if not post_data:
            response.status = http_status.HTTP_400
            response.media = {"error": "Request body not provided"}
            return

        body = json.loads(post_data)

        try:
            errmsg = validate_config(**global_cfg_default(body))
            if errmsg:
                response.status = http_status.HTTP_400
                response.media = {"error": str(errmsg)}
                return
        except TypeError as e:
            response.status = http_status.HTTP_400
            response.media = {"error": str(e)}
            return

        global_cfg_store(body)
        response.status = http_status.HTTP_204
        response.content_type = None


class Key:
    """ API handler for /nom/dop/key/{key} """
    def on_put(self, request, response, key):
        """Keys PUT endpoint.
        ---
        description: Update key name by fingerprint or id.
        parameters:
            - name: key
              in: path
              description: Fingerprint or ID of SSH key
              schema:
                type: string
        responses:
            204:
                description: Successfully updated
            400:
                description: Invalid request
            401:
                description: Could not get username
        """
        post_data = request.bounded_stream.read()
        if not post_data:
            response.status = http_status.HTTP_400
            response.media = {"error": "Request body not provided"}
            return

        body = json.loads(post_data)
        if not body.get("name"):
            response.status = http_status.HTTP_400
            response.media = {"error": "name not provided"}
            return

        name = body.get('name')
        ssh_keys = load_keys_file()

        match_key = [uk for uk in ssh_keys if key in (
            uk['fingerprint'], uk['id'])]
        if match_key:
            # verify if the same name exists for any other key
            if name in [key['name'] for key in ssh_keys if key['id'] != match_key[0]['id']]:
                response.status = http_status.HTTP_400
                response.media = {"error": "name already exists"}
                return
            match_key[0]['name'] = name
        else:
            response.status = http_status.HTTP_400
            response.media = {"error": "{} does not exist".format(key)}
            return

        workspace_store('keys', json.dumps(ssh_keys))

        response.status = http_status.HTTP_204
        response.content_type = None

    def on_delete(self, request, response, key): # pylint: disable=unused-argument
        """Keys DELETE endpoint.
        ---
        description: Delete key by fingerprint or id.
        parameters:
            - name: key
              in: path
              description: Fingerprint or ID of SSH key
              schema:
                type: string
        responses:
            204:
                description: Successfully deleted
            400:
                description: Invalid request
            401:
                description: Could not get username
        """
        ssh_keys = load_keys_file()

        match_key = [(idx, uk) for idx, uk in enumerate(ssh_keys) if key in (uk['fingerprint'], uk['id'])]
        if match_key:
            ssh_keys.pop(match_key[0][0])
        else:
            response.status = http_status.HTTP_400
            response.media = {"error": "{} does not exist".format(key)}
            return

        workspace_store('keys', json.dumps(ssh_keys))

        with authorized_keys_filelock:
            try:
                keys = [key['public_key'] for key in ssh_keys]
                with open(authorized_keys_file, 'w') as fd:
                    fd.write('\n'.join(keys))
            except (IOError, OSError) as err:
                response.status = http_status.HTTP_500
                response.media = {"error": "Failed to delete key from authorized_keys file: {}".format(str(err))}
                # Rollback the key
                ssh_keys.insert(match_key[0], match_key[1])
                workspace_store('keys', json.dumps(ssh_keys))
                return

        response.status = http_status.HTTP_204
        response.content_type = None


class Keys:
    """ API handler for /nom/dop/keys """
    def on_get(self, request, response): # pylint: disable=unused-argument
        """Keys GET endpoint.
        ---
        description: Retrieve SSH keys
        responses:
            200:
                description: List of SSH keys
                content:
                    application/json:
                        schema: SSHKeyList
            401:
                description: Could not get username
        """
        ssh_keys = load_keys_file()
        response.status = http_status.HTTP_200
        response.media = {"ssh_keys": ssh_keys}

    def on_post(self, request, response):
        """Keys POST endpoint.
        ---
        description: Add a new SSH key
        responses:
            201:
                description: Successfully added
            400:
                description: Invalid request
            401:
                description: Could not get username
        """
        post_data = request.bounded_stream.read()
        if not post_data:
            response.status = http_status.HTTP_400
            response.media = {"error": "Request body not provided"}
            return

        body = json.loads(post_data)
        if not body.get("public_key") or not body.get('name'):
            response.status = http_status.HTTP_400
            response.media = {"error": "'name', 'public_key' are required"}
            return

        public_key = body.get("public_key")
        name = body.get("name")

        ssh_keys = load_keys_file()

        if name in [key['name'] for key in ssh_keys]:
            response.status = http_status.HTTP_400
            response.media = {"error": "name already exists"}
            return

        try:
            ssh = sshpubkeys.SSHKey(public_key)
            ssh.parse()
            id = 1 if not ssh_keys else int(ssh_keys[-1]['id'].split('-')[-1]) + 1
            ssh_keys.append({'id': "keys-{}".format(str(id)),
                            'public_key': public_key,
                            'fingerprint': ssh.hash_md5().strip('MD5:'),
                            'name': name})
            workspace_store("keys", json.dumps(ssh_keys))
        except (sshpubkeys.exceptions.InvalidKeyException, UnicodeDecodeError) as err:
            response.status = http_status.HTTP_400
            response.media = {"error": str(err)}
            return

        with authorized_keys_filelock:
            try:
                keys = [key['public_key'] for key in ssh_keys]
                with open(authorized_keys_file, 'w') as fd:
                    fd.write('\n'.join(keys))
            except (IOError, OSError) as err:
                response.status = http_status.HTTP_500
                response.media = {"error": "Failed to save key in authorized_keys file: {}".format(str(err))}
                # Rollback the stored key
                ssh_keys.pop()
                workspace_store("keys", json.dumps(ssh_keys))
                return

        response.status = http_status.HTTP_201
        response.media = ["OK"]
