#!/bin/bash

# This script enables system services required by Secure Provisioning.
# This includes:
# - Enabling the DHCP server on the Management LAN interface
# - Enabling the DNS service
# - Enabling the TFTP service
# - Enabling the NTP service

set -eu

configurator_args=""

enable_dhcp_server () {
	ENABLED=$(config -g config.interfaces.lan.dhcpd.enabled | cut -f2 -d' ')
	if [ "$ENABLED" != "on" ]; then
		# Set a basic DHCP server config, even though it will be overwritten by provisioning
		config -s config.interfaces.lan.dhcpd.enabled=on
		config -s config.interfaces.lan.dhcpd.gateway_self=on
		config -s config.interfaces.lan.dhcpd.pools.total=1

		# Rerun dhcp configurator
		configurator_args="$configurator_args -r dhcp "
	else
		# If already configured, restart to load new config
		if [ -f /var/run/dhcpd.pid ]; then
			kill "$(cat /var/run/dhcpd.pid)" || true
		fi
	fi
}

enable_ntp () {
	ENABLED=$(config -g config.ntp.enabled | cut -f2 -d' ')
	if [ "$ENABLED" != "on" ]; then
		config -s config.ntp.enabled=on
		config -s config.lan.services.ntp.enabled=on
		config -s config.ntp.servers.server1.address=0.au.pool.ntp.org
		config -s config.ntp.servers.server1.key_id=0
		config -s config.ntp.servers.total=1

		# Rerun services configurator
		configurator_args="$configurator_args -r services "
	else
		# If already configured, restart to load new config
		if [ -f /var/run/ntpd.pid ]; then
			kill "$(cat /var/run/ntpd.pid)" || true
		fi
	fi
}

enable_dns () {
	ENABLED=$(config -g config.services.dns.enabled | cut -f2 -d' ')
	if [ "$ENABLED" != "on" ]; then
		config -s config.services.dns.enabled=on

		# If we enabled the service, disable on all non-LAN interfaces
		for interface in wan dialin dialout vpn; do
			config -d "config.$interface.services.dns.enabled"
		done

		# Rerun services configurator
		configurator_args="$configurator_args -r services "
	else
		# If already configured, load new config
		killall -HUP dnsmasq || true
	fi

	# Make sure it is enabled on the LAN interface
	LAN_ENABLED=$(config -g config.lan.services.dns.enabled | cut -f2 -d' ')
	if [ "$LAN_ENABLED" != "on" ]; then
		config -s "config.lan.services.dns.enabled"=on
		config -s "config.lan.services.modified=true"

		# Rerun services configurator
		configurator_args="$configurator_args -r services "
	fi
}

enable_tftp () {
	ENABLED=$(config -g config.services.tftp.enabled | cut -f2 -d' ')
	if [ "$ENABLED" != "on" ]; then
		config -s config.services.tftp.enabled=on

		# If we enabled the service, disable on all non-LAN interfaces
		for interface in wan dialin dialout vpn; do
			config -d "config.$interface.services.tftp.enabled"
		done

		# Rerun services configurator
		configurator_args="$configurator_args -r  services "
	fi

	# Make sure it is enabled on the LAN interface
	LAN_ENABLED=$(config -g config.lan.services.tftp.enabled | cut -f2 -d' ')
	if [ "$LAN_ENABLED" != "on" ]; then
		config -s "config.lan.services.tftp.enabled"=on
		config -s "config.lan.services.modified=true"

		# Rerun services configurator
		configurator_args="$configurator_args -r services "
	fi
}

enable_http () {
	ENABLED=$(config -g config.services.http.enabled | cut -f2 -d' ')
	if [ "$ENABLED" != "on" ]; then
		config -s config.services.http.enabled=on

		# If we enabled the service, disable on all non-LAN interfaces
		for interface in wan dialin dialout vpn; do
			config -d "config.$interface.services.http.enabled"
		done

		# Rerun services configurator
		configurator_args="$configurator_args -r  services "
	else
		# If already configured, load new config
		killall -HUP cherokee || true
	fi

	# Make sure it is enabled on the LAN interface
	LAN_ENABLED=$(config -g config.lan.services.http.enabled | cut -f2 -d' ')
	if [ "$LAN_ENABLED" != "on" ]; then
		config -s "config.lan.services.http.enabled"=on
		config -s "config.lan.services.modified=true"

		# Rerun services configurator
		configurator_args="$configurator_args -r services "
	fi
}

# DHCP server
enable_dhcp_server

# DNS server
enable_dns

# TFTP service
enable_tftp

# HTTP service
enable_http

# NTP service
enable_ntp

# Rerun configurators to apply changes (console servers take care of duplicates passed on cmdline)
# TODO: there are issues with running the configurators on VACM, so don't exit on failure until
# this is fixed.
if [ ! -z "$configurator_args" ]; then
	config $configurator_args || true
fi

exit 0
