#!/bin/bash

# This script checks that the networking configuration supports provisioning,
# and configures a static provisioning address if it doesn't exit.

set -eu

configurators=""

provision_ipaddress_config="$1"
provision_subnet_mask_config="$2"

check_wan_configuration () {
	# If WAN is in an unconfigured state, we can't proceed
	MODE=$(config -g config.interfaces.wan.mode | cut -f2 -d' ')
	if [ -z "$MODE" ]; then
		echo "WAN is unconfigured, exiting"
		return 1
	fi
}

check_network_configuration () {
	# If bridging/bonding mode is enabled, we can't proceed
	BRIDGE_MODE=$(config -g config.system.bridge.enabled | cut -f2 -d ' ')
	if [ "$BRIDGE_MODE" == "on" ]; then
		echo "Network is in bridged mode, exiting"
		return 1
	fi

	BOND_MODE=$(config -g config.system.bond.enabled | cut -f2 -d ' ')
	if [ "$BOND_MODE" == "on" ]; then
		echo "Network is in bonded mode, exiting"
		return 1
	fi
}

configure_mgmt_lan () {
	# If MGMT LAN is enabled in DHCP mode, bail
	MODE=$(config -g config.interfaces.lan.mode | cut -f2 -d' ')
	if [ "$MODE" == "dhcp" ]; then
		echo "Management LAN is configured for DHCP, exiting"
		return 1
	fi

	# If MGMT LAN config doesn't exist or is disabled, enable it with a static address
	LAN_CFG=$(config -g config.interfaces.lan)
	DISABLED=$(config -g config.interfaces.lan.disabled | cut -f2 -d' ')
	if [ -z "$LAN_CFG" ] || [ "$DISABLED" == "on" ]; then
		config -d config.interfaces.lan.disabled
		config -s config.interfaces.lan.mode=static
		config -s config.interfaces.lan.address="${provision_ipaddress_config}"
		config -s config.interfaces.lan.netmask="${provision_subnet_mask_config}"

		# Rerun ipconfig configurator
		configurators="$configurators ipconfig"
	fi
}

# Check state of WAN configuration
check_wan_configuration || exit 1

# Check for unsupported network configurations
check_network_configuration || exit 1

# Management LAN
configure_mgmt_lan || exit 1

# Rerun configurators to apply changes (console servers take care of duplicates passed on cmdline)
# TODO: there are issues with running the configurators on VACM, so don't exit on failure until
# this is fixed.
if [ ! -z "$configurators" ]; then
	config -r ${configurators} || true
fi

# There is a delay between configuring an interface and conman setting the address, so
# give it a while (60s) for the LAN static address to come up if we set it
for i in 1 2 3 4 5 6 7 8 9 10 11 12; do
	if conman_status | grep "network-connection-lan-static up"; then
		break
	fi
	sleep 5
done

exit 0
