#!/bin/bash

log() {
    ( >&2 echo "$@" )
}

logFatal() {
    log "FATAL" "$@"
    exit 1
}

# identify the configured tftp dir on the node (also used for HTTP)
# Note: NetOps files will be stored in a 'nom' subdir instead of the
# root tftp dir.
remoteTFTPDir="$( config -g config.services.tftp.directory )" \
    && remoteTFTPDir="$( echo "$remoteTFTPDir" | cut -f2 -d' ' )" \
    && [ ! -z "$remoteTFTPDir" ] \
    || logFatal "failed to identify the tftp directory"

# Get address of interface to serve ZTP
# - Must not be Docker interface or LHVPN
# - Prefer eth1/net2 static address
# - Fall back to eth0/net1 static address
# TODO: This information should ideally be integrated into device config so that the DHCP configs can be regenerated when addresses are changed etc
# for each interface in order of priority...
for ifaceName in eth2 eth1 eth0; do
    # example of the format required by the playbooks
    remoteInterface="" # eth2
    remoteMac="" # 02:42:71:fe:b7:13
    remoteServer="" # 10.0.0.1
    remoteNetwork="" # 10.0.0.0
    remoteNetmask="" # 255.255.255.0
    remoteNetmaskCIDR="" # 24

    # attempt to load the ipv4 address information
    # e.g: 'inet 10.0.0.1/24 brd 10.0.0.255 scope global net2:static6'
	ifaceDetails="$( ip addr show dev "$ifaceName" 2> /dev/null )" \
	    && ifaceDetails="$( echo "$ifaceDetails" | grep inet )" \
	    && ifaceDetails=( $( echo "$ifaceDetails" | grep -v inet6 ) ) \
	    && [ "${ifaceDetails[0]}" = inet ] \
	    && remoteInterface="${ifaceDetails[$((${#ifaceDetails[@]}-1))]}" \
	    && macDetails="$( ip addr show dev "$remoteInterface" 2> /dev/null )" \
	    && macDetails=( $( echo "$macDetails" | grep link/ether ) ) \
	    && [ "${macDetails[0]}" = link/ether ] \
	    && remoteMac="${macDetails[1]}" \
	    && [ ! -z "$remoteMac" ] \
	    && remoteServer="${ifaceDetails[1]}" \
	    && ipDetails="$( ipcalc --network --netmask --prefix --silent "$remoteServer" )" \
	    && remoteServer="${remoteServer%/*}" \
	    && [ ! -z "$remoteServer" ] \
	    || continue

    while [ ! -z "$ipDetails" ]; do
        lineDelim=$'\n'
        line="${ipDetails##*$lineDelim}"
        if [ "$line" = "$ipDetails" ]; then
            unset ipDetails
        else
            ipDetails="${ipDetails%$lineDelim*}"
        fi
        case "${line%%=*}" in
            NETWORK)
            remoteNetwork="${line#*=}"
            ;;
            NETMASK)
            remoteNetmask="${line#*=}"
            ;;
            PREFIX)
            remoteNetmaskCIDR="${line#*=}"
            ;;
        esac
    done

    if [ -z "$remoteNetwork" ] || [ -z "$remoteNetmask" ] || [ -z "$remoteNetmaskCIDR" ]; then
        continue
    fi

    if [ "${remoteServer}/${remoteNetmaskCIDR}" != "${ifaceDetails[1]}" ]; then
        logFatal "${remoteServer}/${remoteNetmaskCIDR} != ${ifaceDetails[1]}"
    fi

	cat <<EOF
{
  "nom_remote_server":"${remoteServer}",
  "nom_remote_netmask":"${remoteNetmask}",
  "nom_remote_netmask_cidr": "${remoteNetmaskCIDR}",
  "nom_remote_interface":"${remoteInterface}",
  "nom_remote_mac":"${remoteMac}",
  "nom_remote_resource_dir": "${remoteTFTPDir}"
}

EOF

    exit 0
done

logFatal "failed to identify the provisioning network"
