#!/bin/bash

# WARNING the stdout AND stderr of this script is shown on the client performing the push

set -o pipefail || exit 1

# Redirect stdout and stderr of this script to a subshell running an append tee for
# deploy.log, so it's logged to here as well as stdout.
exec &> >( syslog.sh )

# mutually exclusive locking utilising file system atomicity
if ! mkdir /tmp/post_receive.lock; then
	echo "Failed to create lock dir for script"
	exit 1
fi
function cleanup {
	trap - EXIT
	exitCode="$?"
	if ! rmdir /tmp/post_receive.lock; then
		echo "Failed to remove lock dir for resource script"
	fi
	exit "$exitCode"
}
trap cleanup INT TERM EXIT

echo "Deploying resources at $(date)..."

if [ ! -z "$USER_WORKSPACE" ]; then
    sync_workspace.sh "$USER_WORKSPACE" || exit 1
fi

# Not ideal, but we want to keep the auto-commit repo up-to-date, so do a pull whenever central changes
git --git-dir /srv/central-auto/.git --work-tree /srv/central-auto pull origin master &> /dev/null

# Re-clone the repo to ensure the ansible hosts file is up to date, as it will be
# generated in the next step.
rm -rf /srv/checkout \
    && git clone /srv/central /srv/checkout \
    && ( cd /srv/checkout && git-restore-mtime-bare ) \
    || exit 1

echo "Starting resource deployment!"

# Create a directory to deploy from, and resolve the nodes to resources for deployment.
rm -rf /srv/deploy && mkdir -p /srv/deploy/downloads /srv/deploy/scripts || exit 1

echo "Generating deployment from configuration"
generate_deployment || exit 1

# Basic validation of the yaml config.
if ! validate_yaml -d /srv/deploy; then
	echo "Invalid yaml configuration file, cancelling deployment."
	exit 1
fi

# Create the ztp templates before they're deployed within ansible.
generate_ztp_templates || exit 1

# Create the provd configuration file.
generate_provd_conf || exit 1

# WARNING changing any of the echo lines below could break deployment status polling logic
# TODO don't use the last line of a log file as a status code...

echo "Deploying resources to hosts"
ansible-playbook /ansible/deploy.yml -i /etc/scripts/generate_inventory -u root 2>&1
res="$?"

# status code 4 is 'unreachable' nodes - in our case this isn't a failure, the node may
# not be available.
if [[ "$res" != "4" && "$res" != "0" ]]; then
	echo "Deploy failed, check log in Lighthouse central-dop container for details."
	exit 1
fi
echo -e "\n\n"

if [ "$res" = 4 ]; then
	echo "One or more nodes were unreachable. Check all nodes that are in a deployment are connected and have NetOps enabled."
fi

echo "Complete!"
