"""
log provides a class that handles logging DOP-specific messages.
"""
import os

def format_mac(mac):
    """
    format_mac returns the same mac address, adding zero padding and capitalisation
    as necessary. This ensures all stored mac addresses are the standardized.
    """
    octs = mac.split(":")
    if len(octs) != 6:
        return mac
    return "{}:{}:{}:{}:{}:{}".format(
        octs[0].rjust(2, '0'),
        octs[1].rjust(2, '0'),
        octs[2].rjust(2, '0'),
        octs[3].rjust(2, '0'),
        octs[4].rjust(2, '0'),
        octs[5].rjust(2, '0')).upper()


class Log: # pylint: disable=too-many-instance-attributes
    """
    Log contains the logic for printing out the formatted log messages to the
    syslog service.
    """
    def __init__(self, l_type):
        self.log_type = l_type
        self.host = None
        self.mac = None
        self.file = None
        self.script = None
        self.script_stdout = None
        self.timeout = None
        self.exitcode = None
        self.bundle = None
        self.vendor = None
        self.method = None
        self.PREFIX = '[NetOps-DOP] '

    def print_log(self):
        """
        print_log: the main function that constructs a log message and calls
        out to the system's logger command.
        """
        if self.mac is not None:
            if self.log_type == "post_provision_script_stdout":
                self.PREFIX = '[NetOps-DOP device=\\"{}\\" script=\\"{}\\"] '.format(
                    format_mac(self.mac), os.path.basename(self.script))
            else:
                self.PREFIX = '[NetOps-DOP device=\\"{}\\"] '.format(
                    format_mac(self.mac))

        head = self.PREFIX
        body = None
        if self.log_type == "offer":
            body = "Assigned DHCP address of {} to device".format(self.host)
        elif self.log_type == "offer_vendor":
            body = "Received DHCP request from device with vendor ID {}".format(
                self.vendor)
        elif self.log_type == "fileserved":
            body = "Device retrieved resource file {} via {}".format(
                self.file, self.method)
        elif self.log_type == "provision":
            body = "Provisioning device with resource bundle {}".format(
                self.bundle)
        elif self.log_type == "post_provision_script_start":
            body = "Provisioning complete, running post-provisioning script {} with timeout {}".format(
                self.script, self.timeout)
        elif self.log_type == "post_provision_script_complete":
            body = "Post-provisioning script {} completed with exitcode {}".format(
                self.script, self.exitcode)
        elif self.log_type == "post_provision_script_timeout":
            body = "Post-provisioning script {} timed out".format(
                self.script)
        elif self.log_type == "post_provision_script_stdout":
            body = self.script_stdout

        msg = "{}{}".format(head, body)
        os.system("logger {}".format(msg))
