"""
models implements data structures for interacting with ngcs device config
"""
from copy import deepcopy
from typing import Any, Dict, List, Optional

from .util import netmask_cidr


class FirewallServicePort:
    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def data(self) -> Dict[str, Any]:
        return deepcopy(self._data)

    @property
    def protocol(self) -> str:
        return deepcopy(self._data['protocol'])

    @property
    def port(self) -> int:
        return deepcopy(self._data['port'])


class FirewallService:
    """
    /firewall/services items
    example: {"id":"firewall_services-1","ports":[{"protocol":"tcp","port":80},{"protocol":"udp","port":69}],"name":"nom-dop-lan","label":"NetOps Secure Provisioning Module - LAN"}
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def data(self) -> Dict[str, Any]:
        return deepcopy(self._data)

    @property
    def id(self) -> str:
        return deepcopy(self._data['id'])

    @property
    def name(self) -> str:
        return deepcopy(self._data['name'])

    @property
    def ports(self) -> List[FirewallServicePort]:
        return deepcopy([FirewallServicePort(data) for data in self._data['ports']])


class FirewallPolicy:
    """
    /firewall/policies items
    Example JSON:
    {
        "id": "firewall_policies-1",
        "name": "ExamplePolicy",
        "description": "Example",
        "ingress_zones": ["firewall_zones-1"],
        "egress_zones": ["firewall_zones-2"]
    }
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def data(self) -> Dict[str, Any]:
        return deepcopy(self._data)

    @property
    def id(self) -> str:
        return deepcopy(self._data['id'])

    @property
    def name(self) -> str:
        return deepcopy(self._data['name'])

    @property
    def description(self) -> str:
        return deepcopy(self._data['description'])

    @property
    def ingress_zones(self) -> List[str]:
        return deepcopy(self._data['ingress_zones'])

    @property
    def egress_zones(self) -> List[str]:
        return deepcopy(self._data['egress_zones'])


class FirewallZone:
    """
    /firewall/zones items
    example: {"physifs":["system_net_physifs-2"],"name":"wan","label":"WAN","id":"firewall_zones-2","description":"Untrusted connections from the Wide Area Network"}
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def data(self) -> Dict[str, Any]:
        return deepcopy(self._data)

    @property
    def id(self) -> str:
        return deepcopy(self._data['id'])

    @property
    def name(self) -> str:
        return deepcopy(self._data['name'])

    @property
    def interfaces(self) -> List[str]:
        return deepcopy(self._data['physifs'])


class FirewallRule:
    """
    /firewall/rules items
    example: {"id":"firewall_rules-28","service":"nom-dop-lan","zone":"firewall_zones-1"}
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def data(self) -> Dict[str, Any]:
        return deepcopy(self._data)

    @property
    def id(self) -> str:
        return deepcopy(self._data['id'])

    @property
    def service(self) -> str:
        return deepcopy(self._data['service'])

    @property
    def zone(self) -> str:
        return deepcopy(self._data['zone'])


class Interface:
    """
    /physifs items
    example: {"physifs":[{"enabled":true,"id":"system_net_physifs-1","name":"init_net1","media":"ethernet","device":"net1","runtime_status":{"status":"UP","id":"system_net_physifs-1"},"ethernet_setting":{"id":"system_net_physifs-1","link_speed":"auto"},"description":"NET1 - 1G Copper\\/SFP"},{"enabled":true,"id":"system_net_physifs-2","name":"init_net2","media":"ethernet","device":"net2","runtime_status":{"status":"UP","id":"system_net_physifs-2"},"ethernet_setting":{"id":"system_net_physifs-2","link_speed":"auto"},"description":"NET2 - 1G Copper\\/SFP"},{"enabled":true,"id":"system_net_physifs-3","name":"init_net3","media":"ethernet","device":"net3","runtime_status":{"status":"UP","id":"system_net_physifs-3"},"ethernet_setting":{"id":"system_net_physifs-3","link_speed":"auto"},"description":"NET3 - 1G Copper\\/SFP"},{"enabled":false,"id":"system_net_physifs-4","name":"init_cellular","media":"cellular","device":"wwan0","runtime_status":{"status":"NO_SIM","id":"system_net_physifs-4"},"cellular_setting":{"id":"system_net_physifs-4"},"description":"Cellular Interface (LTE)"}]}
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def data(self) -> Dict[str, Any]:
        return deepcopy(self._data)

    @property
    def id(self) -> str:
        return deepcopy(self._data['id'])

    @property
    def enabled(self) -> bool:
        return deepcopy(self._data['enabled'])

    @property
    def device(self) -> str:
        return deepcopy(self._data['device'])

    @property
    def media(self) -> str:
        return deepcopy(self._data['media'])

    @property
    def slaves(self) -> List[str]:
        return deepcopy(self._data.get('slaves', []))


class IPv4StaticSettings:
    """
    /conns item ipv4_static_settings property
    example: {"id":"system_net_conns-1","netmask":"255.255.255.0","address":"192.168.0.1"}
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def netmask(self) -> str:
        return deepcopy(self._data['netmask'])

    @property
    def netmask_cidr(self) -> int:
        return netmask_cidr(self._data['netmask'])

    @property
    def address(self) -> str:
        return deepcopy(self._data['address'])


class Connection:
    """
    /conns items
    example: {"conns":[{"runtime_status":{"status":"reloading","addresses":[{"id":"system_net_conns-1-address-1"}],"last_status_change":1566960684,"id":"system_net_conns-1"},"description":"Default IPv4 Static Address","ipv4_static_settings":{"id":"system_net_conns-1","netmask":"255.255.255.0","address":"192.168.0.1"},"id":"system_net_conns-1","mode":"static","physif":"system_net_physifs-1","name":"default-conn-1"},{"description":"IPv4 DHCP network connection","runtime_status":{"dns1":"192.168.94.1","gateway":"192.168.94.1","id":"system_net_conns-2","addresses":[{"id":"system_net_conns-2-address-1","netmask":"255.255.255.0","address":"192.168.94.234"}],"status":"reloading","last_status_change":1566960684},"id":"system_net_conns-2","mode":"dhcp","physif":"system_net_physifs-1","name":"default-conn-2"},{"description":"IPv6 Auto-configured connection (RA\\/DHCPv6)","runtime_status":{"gateway":" ","id":"system_net_conns-3","addresses":[{"id":"system_net_conns-3-address-1"}],"status":"reloading","last_status_change":1566960684},"id":"system_net_conns-3","mode":"ipv6_automatic","physif":"system_net_physifs-1","name":"v6-dyn-n1-conn"},{"description":"IPv4 DHCP network connection","runtime_status":{"id":"system_net_conns-4","addresses":[{"id":"system_net_conns-4-address-1"}],"status":"reloading","last_status_change":1566960684},"id":"system_net_conns-4","mode":"dhcp","physif":"system_net_physifs-2","name":"v4-dyn-n2-conn"},{"description":"IPv6 Auto-configured connection (RA\\/DHCPv6)","runtime_status":{"id":"system_net_conns-5","addresses":[{"id":"system_net_conns-5-address-1"}],"status":"reloading","last_status_change":1566960684},"id":"system_net_conns-5","mode":"ipv6_automatic","physif":"system_net_physifs-2","name":"v6-dyn-n2-conn"},{"runtime_status":{"status":"reloading","addresses":[{"id":"system_net_conns-6-address-1"}],"last_status_change":1566960684,"id":"system_net_conns-6"},"description":"NetOps Modules Subnet","ipv4_static_settings":{"id":"system_net_conns-6","netmask":"255.255.255.0","address":"10.0.20.1"},"id":"system_net_conns-6","mode":"static","physif":"system_net_physifs-2","name":"v4-nom-n2-conn"}]}
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def data(self) -> Dict[str, Any]:
        return deepcopy(self._data)

    @property
    def id(self) -> str:
        return deepcopy(self._data['id'])

    @property
    def name(self) -> str:
        return deepcopy(self._data['name'])

    @property
    def mode(self) -> str:
        return deepcopy(self._data['mode'])

    @property
    def enabled(self) -> bool:
        if 'runtime_status' not in self._data or not self._data['runtime_status'] or 'status' not in self._data['runtime_status'] or not self._data['runtime_status']['status']:
            return True
        return self._data['runtime_status']['status'] in {'running', 'up'}

    @property
    def interface(self) -> str:
        return deepcopy(self._data['physif'])

    @property
    def ipv4_static_settings(self) -> Optional[IPv4StaticSettings]:
        if 'ipv4_static_settings' not in self._data:
            return None
        return IPv4StaticSettings(deepcopy(self._data['ipv4_static_settings']))

class FailoverToCellular:
    """
    example: {"enabled":true,"service_intercepts":{"ssh":null,"https":null},"mac_address":"00:0f:b0:ca:53:4c","passthrough_physif":"net2"}
    """

    def __init__(self, data: Dict[str, Any]):
        self._data = data

    @property
    def enabled(self) -> bool:
        return deepcopy(self._data['enabled'])

    @property
    def interface(self) -> str:
        return deepcopy(self._data['passthrough_physif'])
