"""
service provides utilities for working with grpc services
"""

import grpc

from typing import (
    Callable,
)


class MiddlewareChannel(grpc.Channel):
    """
    MiddlewareChannel implements grpc channel lifecycle management as falcon middleware
    """

    def __init__(self, factory: Callable[[], grpc.Channel]):  # pylint: disable=super-init-not-called
        self._factory = factory
        self._channel = None

    def _init(self) -> grpc.Channel:
        if self._channel is None:
            self._channel = self._factory()
        return self._channel

    def subscribe(self, callback, try_to_connect=False):
        self._init().subscribe(callback, try_to_connect)

    def unsubscribe(self, callback):
        if self._channel is not None:
            self._channel.unsubscribe(callback)

    def unary_unary(self, method, request_serializer=None, response_deserializer=None):
        return lambda *args, **kwargs: self._init().unary_unary(method, request_serializer, response_deserializer)(*args, **kwargs)

    def unary_stream(self, method, request_serializer=None, response_deserializer=None):
        return lambda *args, **kwargs: self._init().unary_stream(method, request_serializer, response_deserializer)(*args, **kwargs)

    def stream_unary(self, method, request_serializer=None, response_deserializer=None):
        return lambda *args, **kwargs: self._init().stream_unary(method, request_serializer, response_deserializer)(*args, **kwargs)

    def stream_stream(self, method, request_serializer=None, response_deserializer=None):
        return lambda *args, **kwargs: self._init().stream_stream(method, request_serializer, response_deserializer)(*args, **kwargs)

    def close(self):
        if self._channel is not None:
            self._channel.close()
            self._channel = None

    def __enter__(self):
        raise NotImplementedError()

    def __exit__(self, exc_type, exc_val, exc_tb):
        raise NotImplementedError()

    def process_response(self, req, resp, resource, req_succeeded):  # pylint: disable=unused-argument
        self.close()
