"""
This module provides REST handlers for Network Access Policies.
"""
import falcon.status_codes as http_status
from central_sdi import policies
from central_sdi.api import schema


class Policies:
    """
    Handlers for creating, reading, editing and deleting policies.
    """
    schema = schema.PolicyList()

    def on_get(self, request, response):  # pylint: disable=unused-argument
        """
        Policies GET endpoint.
        ---
        description: Retrieve all of the policies, and return them as a list.
        responses:
            200:
                description: Successfully retrieved
                content:
                    application/json:
                        schema: PolicyList
        """
        request.context.result = policies.load()

    def on_put(self, request, response):
        """
        Policies PUT endpoint.
        ---
        description: Save or update the list of policies.
        requestBody:
            content:
                application/json:
                    schema: PolicyList
        responses:
            204:
                description: Successfully updated
            400:
                description: Invalid request
        """
        # Validate the request
        groups = []
        for policy in request.context.json["policies"]:
            # At the current stage, only support one group
            if len(policy["groups"]) != 1:
                response.status = http_status.HTTP_400
                response.media = {"error": "Invalid request - One policy should be only for one group"}
                return
            group = policy["groups"][0]
            # Check if policy for this group has beed defined before
            if group in groups:
                response.status = http_status.HTTP_400
                response.media = {"error": f"Invalid request - Multiple policies for group '{group}'"}
                return
            # Add to the groups list to check if the group has been defined before
            groups.append(group)

            zones = []
            # validate zones value
            if len(policy["zones"]) < 1:
                response.status = http_status.HTTP_400
                response.media = {"error": f"Invalid request - No valid zone provide for group: '{group}'"}
                return
            # Check if there are duplicated zones
            for zone in policy["zones"]:
                if zone in zones:
                    # Check if this group has been defined before
                    response.status = http_status.HTTP_400
                    response.media = {"error": f"Invalid request - Duplicated zones '{zone}' defined for group: '{group}'"}
                    return
                # Add to the groups list to check if the group has been defined before
                zones.append(zone)

        policies.store(request.context.json)
        response.status = http_status.HTTP_204
