""" Authorize http header for a request
"""
from central_sdi.api import authz, errors, router


class AuthenticationMiddleware:
    """
    AuthenticationMiddleware: a Falcon class that is called by the API when
    processing requests.
    """

    def process_resource(self, request, response, resource, params):  # pylint: disable=unused-argument
        """process_resource: Called after routing has occured but before handler
        is called.
        """
        # Auth required
        root = router.Root()
        if root.auth_required(request.uri_template):
            scheme, token = get_http_authorization(request)
            if scheme == "Cert":
                # Cert auth is not implemented yet
                raise errors.Forbidden(error="Permission denied")

            if token is None:
                raise errors.Forbidden(error="Permission denied")

            request.context["token"] = token
            secctxt = authz.secctxt_from_token(token)
            request.context["secctxt"] = secctxt

            if not authz.can(token, f"/nom/sdi{request.path}", request.method):
                raise errors.Forbidden(error="Permission denied")


def get_http_authorization(request):

    """ Get the token from the header and validate
    """

    if request.env.get('SSL_CLIENT_VERIFY') == "SUCCESS":
        return "Cert", None

    if request.get_header("Token") is not None:
        return "Token", request.get_header("Token")

    if request.auth is None:
        return None, None

    http_auth_trimmed = request.auth.strip()

    components = http_auth_trimmed.split()
    if len(components) != 2:
        # not valid authorization header
        return None, None
    req_scheme, req_cred = components

    if req_scheme == "Token":
        return "Token", req_cred

    return None, None
