"""
deployment: handle the resolution of nodes-to-resources for a single deployment.
The deployment process must handle order of precedence for resources and nodes.
Currently, the order is (where 1 is least preferred):
    1: git (/srv/checkout)
    2: LighthouseAdmin (/srv/central-ui/root)
    3: NodeAdmin (/srv/central-ui/* - other directories excluding root)

The deployment needs to determine, at each level of preference, what resources
each node will receive from the deployment. If the same node exists in a deployment
on the next preference level, the resources the node receives will be only those
specified in the higher preference level.

When a smartgroup has been specified, the nodes belonging to that smartgroup are
resolved at the time of deployment. This ensures that the outcome of this script
is what is actually deployed.

Before updating this script, consider carefully about how it will modify the
deployment behaviour!
"""
import errno
import os
import yaml
from glob import glob

import central_dop.config
from central_dop.api import dop
from central_dop.nodes import resolve_smartgroup_to_ids

SMARTGROUP_CACHE = {}


class Resource:
    """
    Resource represents a resource bundle from a config file.
    """

    def __init__(self, identifier, directory, priority, config, snippet=None, downloads=None):  # pylint: disable=too-many-arguments
        self._identifier = str(identifier)
        self._directory = directory
        self._priority = priority
        self._config = config
        self._snippet = snippet
        self._downloads = downloads

    def __hash__(self):
        return hash(self.identifier)

    def __eq__(self, other):
        return self.identifier == other.identifier

    @property
    def identifier(self):
        """
        the resource bundle name for the Resource
        """
        return self._identifier

    @property
    def directory(self):
        """
        the directory this config was sourced from.
        """
        return self._directory

    @property
    def config(self):
        """
        the configuration for a given Resource
        """
        return self._config

    @property
    def priority(self):
        """
        the priority level of this Resource, so we can
        use it to determine which bundles should end up being deployed to the
        Node the Resource belongs to.
        """
        return self._priority

    @property
    def snippet(self):
        """
        the directory path for user defined dhcp snippets
        """
        return self._snippet

    @property
    def has_snippet(self):
        """
        quick check for user defined snippets
        """
        return self.snippet is not None

    @property
    def downloads(self):
        """
        the directory path for user defined download directory
        """
        return self._downloads

    @property
    def has_downloads(self):
        """
        true if the resource has downloads specified
        """
        return self.downloads is not None


class Node:
    """
    Node represents a Node from a config file.
    """

    def __init__(self, identifier, global_config=None):
        self._identifier = str(identifier)
        self._resources = []
        self._global_config = global_config

    def __hash__(self):
        return hash(self.identifier)

    def __eq__(self, other):
        return self.identifier == other.identifier

    @property
    def identifier(self):
        """
        the node id for this Node.
        """
        return self._identifier

    @property
    def resources(self):
        """
        the Resources associated with this Node.
        """
        return self._resources

    @property
    def ansible_vars(self):
        """
        ansible_vars are mapped to per node and propagated as host vars
        :return:
        """
        return {
            'provision_ipaddress_config': self._global_config['ipaddress'],
            'provision_subnet_mask_config': self._global_config['subnet_mask'],
        }

    def add_resources(self, *resources):
        """
        add an array of Resource objects to this Node.
        """
        self.resources.extend(*resources)

    def get_highest_priority_resources(self):
        """
        return only the highest priority Resources associated with this Node.
        """
        priorities = [resource.priority for resource in self.resources]
        if not priorities:
            return []

        highest_priority = max(0, *priorities)
        return [resource
                for resource in self.resources
                if resource.priority == highest_priority]


def enumerate_config_directories(include_dirs, search_dir):
    """
    looks for all directories that we'll be sourcing configuration
    files from. The directories are sorted by most recently modified.
    @include_dirs are the directories that should always be included
    @search_dir is the start of a directory to search based on modified time
    """
    deploy_dirs = []
    deploy_dirs.extend(include_dirs)

    user_directories = [user_dir for user_dir in os.listdir(
        search_dir) if user_dir != "root"]
    if not user_directories:
        return deploy_dirs

    directories_modified = [
        (int(os.path.getmtime(os.path.join(search_dir, file))),
         os.path.join(search_dir, file))
        for file in user_directories
    ]
    directories_sorted = [path[1] for path in sorted(directories_modified)]

    deploy_dirs.extend(directories_sorted)
    return deploy_dirs


def get_all_nodes_in_smartgroup(smartgroup):
    """
    pulls the list of nodes in a given smartgroup. results are cached between
    function calls.
    """
    global SMARTGROUP_CACHE  # pylint: disable=global-variable-not-assigned
    if smartgroup in SMARTGROUP_CACHE:
        return SMARTGROUP_CACHE[smartgroup]

    try:
        SMARTGROUP_CACHE[smartgroup] = resolve_smartgroup_to_ids(smartgroup)
    except central_dop.nodes.Error as err:
        print("error resolving smartgroup: {}".format(err.message))

        # ignore error as user has provided invalid smartgroup
        SMARTGROUP_CACHE[smartgroup] = []

    return SMARTGROUP_CACHE[smartgroup]


def link(s, d):
    """
    wraps the symlinking of s to d to catch errors. link will create any
    destination directories that do not exist already.
    @s - the filepath to the source file
    @d - the filepath to the destination file
    """
    try:
        os.makedirs(os.path.dirname(d))
    except FileExistsError:
        pass

    try:
        os.symlink(s, d)
    except OSError as err:
        if err.errno != errno.EEXIST:
            raise err


def link_to_dir(source, destination, is_pattern=False):
    """
    links files from one directory to another.
    @source - the file/directory to source the files to be linked
    @destination - the directory to point the links
    @is_pattern - indicates whether @source is a directory to search in,
        a glob pattern to use to find files in a directory.
    """

    if is_pattern:
        for file in glob("{}".format(source)):
            link(file, "{}/{}".format(destination, os.path.basename(file)))
    else:
        for file in glob("{}/*".format(source)):
            link(file, "{}/{}".format(destination, os.path.basename(file)))


def generate_config(output_dir, include_dirs, search_dir):
    """
    this handles:
        1 finding all deployment configurations for all users
        2 determining the resources for the nodes from the combined output
        3 symlinking the resources to the canonical deployment directory
        4 writing out the final deployment config to disk

    @output_dir - the directory to save the generated output
    @include_dirs - the directories to include in the config sourcing
    @search_dir - the root for searching for user configs
    """

    deployment = {}
    settings = None

    # get the global config, to serve as a base for per-node config
    # TODO the package structure is looking pretty nonsensical and is overdue for a refactor or five
    global_config = dop.global_cfg_default(dop.global_cfg_load())

    # The priority score is used to rank what resource bundles should be
    # deployed to each of the nodes based on the directory the config was found in.
    priority = 0

    dirs = enumerate_config_directories(include_dirs, search_dir)
    for directory in dirs:
        try:
            config = central_dop.config.load_config_from_dir(directory)
        except central_dop.config.InputError:
            # suppress error as directory may not contain config
            continue
        except central_dop.config.ResourceError as err:
            print("Invalid config: {}".format(err))
            continue

        # References to the actual configuration from the user.
        inventory_list = config["node_inventory"]
        resource_list = config["device_resources"]
        deploy_list = config["deployment"]
        deploy_inventories = deploy_list.keys()

        for inv in deploy_inventories:
            if not inv in inventory_list:
                print("Invalid inventory {} found in deployment list.".format(inv))
                continue

            # Resolve all of the nodes for this inventory, either by static list
            # or smartgroup.
            nodes = []
            inventory = inventory_list[inv]
            for kind in inventory:
                if kind == 'smartgroup':
                    nodes = get_all_nodes_in_smartgroup(
                        inventory_list[inv][kind])
                elif kind == 'static':
                    nodes = inventory_list[inv][kind]

            # Now we need to determine the resource bundles destined to deploy
            # to this inventory. Once we have the resources, we can associate
            # the resources with the nodes.
            deploy_resources = []
            inventory_resources = config["deployment"][inv]
            for resource in inventory_resources:
                if not resource in resource_list:
                    print("Invalid resource found in deployment list.")
                    continue

                # Check for any user defined dhcpd or download directory
                # snippets which are specified by node inventory. We need to do
                # this here as it's the last time we know what inventory the
                # nodes belong to.
                snippet = None
                source = "{}/{}/dhcpd".format(directory, inv)
                if os.path.isdir(source):
                    snippet = source

                downloads = None
                source = "{}/{}/downloads".format(directory, inv)
                if os.path.isdir(source):
                    downloads = source

                deploy_resources.append(
                    Resource(resource, directory, priority, resource_list[resource], snippet, downloads))

            # Save the node_id -> Node mapping to the master table.
            for node_id in nodes:
                if node_id not in deployment:
                    deployment[node_id] = Node(node_id, global_config=global_config)
                deployment[node_id].add_resources(deploy_resources)

        # If user has specified settings in their config, update the main deployment settings.
        # This means the user with the highest priority will override settings defined
        # by all other users.
        if 'settings' in config:
            settings = config['settings']

        priority += 1

    # We've determined the config that will be deployed for each name. Setup the
    # files for distribution, and generate the final yaml config that will be used
    # by ansible for deployment.
    ansible_config = {
        'deployment': {},
        'node_inventory': {},
        'device_resources': {},
        'node_vars': {},
    }

    if settings:
        ansible_config['settings'] = settings

    for _, node in deployment.items():
        # Create a node inventory for just this node, and add the node to the
        # deployment list, so we can append each of the bundles to be deployed.
        node_inv = "{}".format(node.identifier).replace("-", "_")
        ansible_config['node_inventory'][node_inv] = {
            'static': [
                node.identifier,
            ]
        }
        ansible_config['deployment'][node_inv] = []
        ansible_config['node_vars'][node_inv] = node.ansible_vars

        for resource in node.get_highest_priority_resources():
            directory = resource.directory

            # Link any files required by the resource directly into the node
            # inventory directory.
            for res_file in ['config_file', 'image_file', 'script_file']:
                if res_file in resource.config:
                    f = resource.config.get(res_file)
                    source = "{}/downloads/{}".format(directory, f)
                    destination = "{}/{}/downloads/{}".format(
                        output_dir, node_inv, f)
                    link(source, destination)

            # Then check the scripts directory.
            source = "{}/scripts/".format(directory)
            destination = "{}/scripts/".format(output_dir)
            link_to_dir(source, destination)

            # We also support host-specific dhcp snippets and files, so symlink
            # any of those too.
            source = "{}/[0-9]*".format(directory)
            destination = output_dir
            link_to_dir(source, destination, True)

            if resource.has_snippet:
                source = "{}/*".format(resource.snippet)
                destination = "{}/{}/dhcpd".format(output_dir, node_inv)
                link_to_dir(source, destination, True)

            if resource.has_downloads:
                source = "{}/*".format(resource.downloads)
                destination = "{}/{}/downloads".format(output_dir, node_inv)
                link_to_dir(source, destination, True)

            # Create a resource bundle for all of the resources for the node.
            bundle_name = resource.identifier
            ansible_config['device_resources'][bundle_name] = resource.config

            # Add the node inventory to the final list of deployments.
            ansible_config['deployment'][node_inv].append(bundle_name)

    with open(f"{output_dir}/config.yml", "w") as file:
        yaml.dump(ansible_config, file)
