#!/usr/bin/python3
"""
authz manages the authorization initialisation for a given authentication token
to control acess rights for resources.
"""
import json
import os
from enum import IntEnum, auto
import requests
import grpc
from central_dop.nodes import get_api_version, get_host_address
from netops import authz

requests.packages.urllib3.disable_warnings()  # pylint: disable=no-member


class Verb(IntEnum):
    """ Verb constructs the available resource access methods."""
    READ = auto()
    WRITE = auto()
    CREATE = auto()
    DELETE = auto()
    EXECUTE = auto()
    UNKNOWN = auto()


class Right(IntEnum):
    """Right is a list of all resources and their corresponding verbs."""
    RIGHT_VIEW_DEVICE_RESOURCES = 0
    RIGHT_MODIFY_DEVICE_RESOURCES = 1
    RIGHT_CREATE_DEVICE_RESOURCES = 2
    RIGHT_DELETE_DEVICE_RESOURCES = 3
    RIGHT_VIEW_NODE_INVENTORY = 4
    RIGHT_MODIFY_NODE_INVENTORY = 5
    RIGHT_CREATE_NODE_INVENTORY = 6
    RIGHT_DELETE_NODE_INVENTORY = 7
    RIGHT_CREATE_DYNAMIC_NODE_INVENTORY = 8
    RIGHT_VIEW_ROUTES = 9
    RIGHT_VIEW_RIGHTS = 10
    RIGHT_VIEW_RESOURCE_PUSH_STATUS = 11
    RIGHT_CREATE_RESOURCE_PUSH = 12
    RIGHT_VIEW_RESOURCE_SYNC_STATUS = 13
    RIGHT_CREATE_RESOURCE_SYNC = 14
    RIGHT_VIEW_SETTINGS = 15
    RIGHT_MODIFY_SETTINGS = 16
    RIGHT_DEPLOY_SMARTGROUP = 17  # This right grants access to deploy resources to a smartgroup
    RIGHT_CREATE_SSH_KEYS = 18
    RIGHT_VIEW_SSH_KEYS = 19
    RIGHT_MODIFY_SSH_KEYS = 20
    RIGHT_DELETE_SSH_KEYS = 21


# Role->Rights definitions
lh_admin_global_rights = [
    Right.RIGHT_VIEW_DEVICE_RESOURCES,
    Right.RIGHT_MODIFY_DEVICE_RESOURCES,
    Right.RIGHT_CREATE_DEVICE_RESOURCES,
    Right.RIGHT_DELETE_DEVICE_RESOURCES,
    Right.RIGHT_VIEW_NODE_INVENTORY,
    Right.RIGHT_MODIFY_NODE_INVENTORY,
    Right.RIGHT_CREATE_NODE_INVENTORY,
    Right.RIGHT_DELETE_NODE_INVENTORY,
    Right.RIGHT_CREATE_DYNAMIC_NODE_INVENTORY,
    Right.RIGHT_VIEW_ROUTES,
    Right.RIGHT_VIEW_RIGHTS,
    Right.RIGHT_VIEW_RESOURCE_PUSH_STATUS,
    Right.RIGHT_CREATE_RESOURCE_PUSH,
    Right.RIGHT_VIEW_RESOURCE_SYNC_STATUS,
    Right.RIGHT_CREATE_RESOURCE_SYNC,
    Right.RIGHT_VIEW_SETTINGS,
    Right.RIGHT_MODIFY_SETTINGS,
    Right.RIGHT_DEPLOY_SMARTGROUP,
    Right.RIGHT_CREATE_SSH_KEYS,
    Right.RIGHT_MODIFY_SSH_KEYS,
    Right.RIGHT_DELETE_SSH_KEYS,
    Right.RIGHT_VIEW_SSH_KEYS
]
lh_admin_sg_rights = []

node_admin_global_rights = [
    Right.RIGHT_VIEW_DEVICE_RESOURCES,
    Right.RIGHT_MODIFY_DEVICE_RESOURCES,
    Right.RIGHT_CREATE_DEVICE_RESOURCES,
    Right.RIGHT_DELETE_DEVICE_RESOURCES,
    Right.RIGHT_VIEW_NODE_INVENTORY,
    Right.RIGHT_MODIFY_NODE_INVENTORY,
    Right.RIGHT_CREATE_NODE_INVENTORY,
    Right.RIGHT_DELETE_NODE_INVENTORY,
    Right.RIGHT_CREATE_DYNAMIC_NODE_INVENTORY,
    Right.RIGHT_VIEW_ROUTES,
    Right.RIGHT_VIEW_RIGHTS,
    Right.RIGHT_VIEW_RESOURCE_PUSH_STATUS,
    Right.RIGHT_CREATE_RESOURCE_PUSH,
    Right.RIGHT_VIEW_RESOURCE_SYNC_STATUS,
    Right.RIGHT_CREATE_RESOURCE_SYNC,
]
node_admin_sg_rights = [
    Right.RIGHT_DEPLOY_SMARTGROUP
]

node_user_global_rights = []
node_user_sg_rights = []


def add_rights_for_role(role, rights, scope):
    """Add all a given role's rights to the rights list"""
    target_rights = []

    # this could be flattened to a single dictionary lookup
    # probably needs a general re{jig,architecture} in the context of LiPy
    if scope == "global":
        if role == "LighthouseAdmin":
            target_rights = lh_admin_global_rights
        elif role == "NodeAdmin":
            target_rights = node_admin_global_rights
        elif role == "NodeUser":
            target_rights = node_user_global_rights
    else:
        if role == "LighthouseAdmin":
            target_rights = lh_admin_sg_rights
        elif role == "NodeAdmin":
            target_rights = node_admin_sg_rights
        elif role == "NodeUser":
            target_rights = node_user_sg_rights

    for r in target_rights:
        rights[r] = True


def add_global_rights(rights, groups):
    """ add global rights to all groups according to the group type """
    for group in groups['groups']:
        if not group['enabled']:
            continue

        if group['mode'] == 'global' and 'global_roles' in group:
            add_rights_for_role("LighthouseAdmin", rights, "global")
        elif group['mode'] == 'smart_group':
            if group['smart_group_roles'] == "NodeAdmin":
                add_rights_for_role("NodeAdmin", rights, "global")
            elif group['smart_group_roles'] == 'NodeUser':
                add_rights_for_role("NodeUser", rights, "global")


def string_to_verb(verb_str):
    """ convert the given verb to the authz enum """
    verb = Verb.UNKNOWN
    if verb_str == "READ":
        verb = Verb.READ
    elif verb_str == "WRITE":
        verb = Verb.WRITE
    elif verb_str == "CREATE":
        verb = Verb.CREATE
    elif verb_str == "DELETE":
        verb = Verb.DELETE
    elif verb_str == "EXECUTE":
        verb = Verb.EXECUTE
    return verb


############# Caching LH API functions ##################

def get_cached_groups(token):
    """Retrieve list of groups for a session. If cached results exist,
    return those. If not, retrieve them from Lighthouse API and cache them
    for next time.
    Note: The cache has the same lifetime as the session token, which mirrors
    the LH behavior where a user needs to log out for authz changes to apply."""
    token_cache_dir = "/tmp/.authz_cache/{}".format(token)

    if not os.path.isdir(token_cache_dir):
        os.makedirs(token_cache_dir)

    group_file = "{}/groups".format(token_cache_dir)
    if os.path.isfile(group_file):
        with open(group_file) as f:
            return json.loads(f.readline())

    headers = {"Authorization": "Token {}".format(token)}
    hostname = get_host_address()
    g = requests.get("https://{}/api/{}/groups".format(hostname, get_api_version()), headers=headers, verify=False)
    if g.status_code != 200:
        return None

    with open(group_file, "w+") as f:
        f.write(g.text)

    return g.json()


class BaseEntityHandler:
    """This is the base class for entity handlers. It sets up rights on initialization,
    and stores the entity. Each entity will be handled by an extendsion of this class"""

    def __init__(self, entity, lh_groups):
        self._rights = [False] * len(Right)
        self._entity = entity
        self._lh_groups = lh_groups
        add_global_rights(self._rights, lh_groups)

    @property
    def rights(self):
        """get all rights for the handler"""
        return self._rights

    @property
    def entity(self):
        """ get the corresponding entity for the handler"""
        return self._entity

    @property
    def lh_groups(self):
        """ get all groups for the handler"""
        return self._lh_groups

    def can(self, verb):
        """
        Virtual function to resolve authz for the given entity/verb combo.
        Implemented in each of the extended classes.
        """


def new_handler_for_entity(entity, lh_groups):
    """ retrieve the entity specific handler for the token"""
    handler = None
    if entity == "/nom/dop/device_resources_files":
        handler = DeviceResourceFilesEntityHandler(entity, lh_groups)
    elif entity.startswith("/nom/dop/device_resources"):
        handler = DeviceResourceEntityHandler(entity, lh_groups)
    elif entity == "/nom/dop/node_inventory/dynamic":
        handler = DynamicNodeInventoryEntityHandler(entity, lh_groups)
    elif entity == "/nom/dop/node_inventory/static":
        handler = StaticNodeInventoryEntityHandler(entity, lh_groups)
    elif entity.startswith("/nom/dop/node_inventory"):
        handler = NodeInventoryEntityHandler(entity, lh_groups)
    elif entity == "/nom/dop/push":
        handler = PushEntityHandler(entity, lh_groups)
    elif entity == "/nom/dop/rights":
        handler = RightsEntityHandler(entity, lh_groups)
    elif entity == "/nom/dop/routes":
        handler = RoutesEntityHandler(entity, lh_groups)
    elif entity == "/nom/dop/settings":
        handler = SettingsEntityHandler(entity, lh_groups)
    elif entity == "/nom/dop/sync":
        handler = SyncEntityHandler(entity, lh_groups)
    elif entity.startswith("/nom/dop/smartgroup"):
        handler = SmartgroupEntityHandler(entity, lh_groups)
    elif entity.startswith("/nom/dop/config"):
        handler = SettingsEntityHandler(entity, lh_groups)
    elif entity.startswith("/nom/dop/keys"):
        handler = KeysEntityHandler(entity, lh_groups)
    return handler


class DeviceResourceEntityHandler(BaseEntityHandler):
    """handler for all device resource actions"""

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        grant_access = False
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_DEVICE_RESOURCES]:
            grant_access = True
        elif verb == Verb.WRITE and self.rights[Right.RIGHT_MODIFY_DEVICE_RESOURCES]:
            grant_access = True
        elif verb == Verb.CREATE and self.rights[Right.RIGHT_CREATE_DEVICE_RESOURCES]:
            grant_access = True
        elif verb == Verb.DELETE and self.rights[Right.RIGHT_DELETE_DEVICE_RESOURCES]:
            grant_access = True
        return grant_access


class DeviceResourceFilesEntityHandler(DeviceResourceEntityHandler):
    """This handler uses the same rights as DeviceResourceEntityHandler so we
    can just extend that"""


class DynamicNodeInventoryEntityHandler(BaseEntityHandler):
    """handler for the dynamic node inventories"""

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        grant_access = False
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_NODE_INVENTORY]:
            grant_access = True
        elif verb == Verb.WRITE and self.rights[Right.RIGHT_MODIFY_NODE_INVENTORY]:
            grant_access = True
        elif verb == Verb.CREATE and self.rights[Right.RIGHT_CREATE_DYNAMIC_NODE_INVENTORY]:
            grant_access = True
        elif verb == Verb.DELETE and self.rights[Right.RIGHT_DELETE_NODE_INVENTORY]:
            grant_access = True
        return grant_access

class KeysEntityHandler(BaseEntityHandler):
    """ handler for the ssh keys """

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        grant_access = False
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_SSH_KEYS]:
            grant_access = True
        elif verb == Verb.WRITE and self.rights[Right.RIGHT_MODIFY_SSH_KEYS]:
            grant_access = True
        elif verb == Verb.CREATE and self.rights[Right.RIGHT_CREATE_SSH_KEYS]:
            grant_access = True
        elif verb == Verb.DELETE and self.rights[Right.RIGHT_DELETE_SSH_KEYS]:
            grant_access = True
        return grant_access


class NodeInventoryEntityHandler(BaseEntityHandler):
    """handler for the regular node inventories"""

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        grant_access = False
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_NODE_INVENTORY]:
            grant_access = True
        elif verb == Verb.WRITE and self.rights[Right.RIGHT_MODIFY_NODE_INVENTORY]:
            grant_access = True
        elif verb == Verb.CREATE and self.rights[Right.RIGHT_CREATE_NODE_INVENTORY]:
            grant_access = True
        elif verb == Verb.DELETE and self.rights[Right.RIGHT_DELETE_NODE_INVENTORY]:
            grant_access = True
        return grant_access


class StaticNodeInventoryEntityHandler(NodeInventoryEntityHandler):
    """This handler uses the same rights as NodeInventoryEntityHandler so we
    can just extend that"""


class RightsEntityHandler(BaseEntityHandler):
    """handler for the user rights"""

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_RIGHTS]:
            return True
        return False


class RoutesEntityHandler(BaseEntityHandler):
    """handler for the user routes"""

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_ROUTES]:
            return True
        return False


class PushEntityHandler(BaseEntityHandler):
    """ handler for the resource push functions """

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        grant_access = False
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_RESOURCE_PUSH_STATUS]:
            grant_access = True
        elif verb == Verb.CREATE and self.rights[Right.RIGHT_CREATE_RESOURCE_PUSH]:
            grant_access = True
        return grant_access


class SettingsEntityHandler(BaseEntityHandler):
    """ handler for the config settings """

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        grant_access = False
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_SETTINGS]:
            grant_access = True
        elif verb == Verb.WRITE and self.rights[Right.RIGHT_MODIFY_SETTINGS]:
            grant_access = True
        return grant_access


class SyncEntityHandler(BaseEntityHandler):
    """ handler for the synchronisation actions """

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""
        grant_access = False
        if verb == Verb.READ and self.rights[Right.RIGHT_VIEW_RESOURCE_SYNC_STATUS]:
            grant_access = True
        elif verb == Verb.CREATE and self.rights[Right.RIGHT_CREATE_RESOURCE_SYNC]:
            grant_access = True
        return grant_access


class SmartgroupEntityHandler(BaseEntityHandler):
    """ handler for the config settings """

    def can(self, verb):
        """true if given handler has @verb in list of rights for the entity"""

        if verb == Verb.EXECUTE and self.rights[Right.RIGHT_DEPLOY_SMARTGROUP]:
            return True

        # If this is the root smartgroups entity, return true here as they will be authorized
        # individually further down.
        if self.entity == "/nom/dop/smartgroups":
            return True

        # specific smartgroups will be the last item in the entity string.
        split_entity = self.entity.split("/")
        if len(split_entity) != 5:
            return False

        # Now add any smartgroup-scoped rights for any groups linked to it.
        smartgroup = split_entity[4]
        for group in self.lh_groups['groups']:
            if not group['enabled']:
                continue

            if group['mode'] == 'smart_group' and group['smart_group'] == smartgroup:
                if group['smart_group_roles'] == "NodeAdmin":
                    add_rights_for_role("NodeAdmin", self.rights, "smart_group")
                elif group['smart_group_roles'] == 'NodeUser':
                    add_rights_for_role("NodeUser", self.rights, "smart_group")

        if verb == Verb.EXECUTE and self.rights[Right.RIGHT_DEPLOY_SMARTGROUP]:
            return True

        return False


def main_handler(token, entity, verb):
    """
    Takes a token, entity and verb, and returns a boolean indicating if authz
    are permitted for the token.
    """
    if isinstance(verb, str):
        verb_enum = string_to_verb(verb)
    else:
        verb_enum = verb

    if verb_enum == Verb.UNKNOWN:
        return False

    try:
        with authz.grpc_channel() as channel:
            lh_groups = authz.dummy_groups(channel, token)
    except grpc.RpcError as err:
        if err.code() != grpc.StatusCode.UNIMPLEMENTED:  # pylint: disable=no-member
            raise err
        lh_groups = get_cached_groups(token)
    if not lh_groups:
        return False

    handler = new_handler_for_entity(entity, lh_groups)
    if handler is None:
        return False

    return handler.can(verb_enum)
