"""
target_device provides a class that holds information on devices being
provisioned by the remote-dop container. On initialisation, a handle to the
redis database is required.
"""
import json

KEY_PREFIX = "device"
def _key(suff):
    """ key is the formatted key for the object to be saved in redis """
    return "{}:{}".format(KEY_PREFIX, suff)

# from_mac and from_ip wrap the get function with a friendly name API.
def from_mac(td, mac):
    """
    from_mac is used to retrieve a TargetDevice's data from the
    database by the given mac address.
    """
    return get(td, mac)

def from_ip(td, ip):
    """
    from_ip is used to retrieve a TargetDevice's data from the
    database by the given IP address.
    """
    return get(td, ip)

def get(td, k):
    """
    Searches the handle associated with the TargetDevice for a particular device.

    @td is a TargetDevice object with an associated handle.

    @k is the key to search for, which should be 'device:<mac>' or 'device:<ip>'.
    """
    t = td.handle.get(_key(k))
    if t is None:
        return td
    return td.from_json(t)

class Error(Exception):
    """Base class for exceptions in the TargetDevice module."""
    def __init__(self, msg): # pylint: disable=super-init-not-called
        self._message = msg

    @property
    def message(self):
        """ returns the message of this particular exception """
        return self._message

class LoadError(Error):
    """
    LoadError is raised when the json provided was not valid.
    """

class TargetDevice: # pylint: disable=too-many-instance-attributes
    """
    TargetDevice is a class wrapping the serialization of a target device to and
    from the local Redis database. This is used across various packages in the
    remote-dop container.
    """
    def __init__(self, db_handle):
        self.handle = db_handle
        self._mac = None
        self._ip = None
        self._hostname = None
        self._bundles = []
        self._vendor = None
        self._files_downloaded = []
        self._provisioned = False
        self._post_provision_scripts_run = []

    @property
    def provisioned(self):
        """ true if the device has retrieved all required files """
        return self._provisioned

    @provisioned.setter
    def provisioned(self, value):
        """ set to true if the device has retrieved all required files """
        self._provisioned = value

    @property
    def post_provision_scripts_run(self):
        """ return list of post-provisioning scripts that have been run """
        return self._post_provision_scripts_run

    @post_provision_scripts_run.setter
    def post_provision_scripts_run(self, value):
        """ set list of post-provisioning scripts that have been run """
        self._post_provision_scripts_run = value

    @property
    def mac(self):
        """ the mac address of a target device """
        return self._mac

    @mac.setter
    def mac(self, value):
        """ set the mac address of a target device """
        self._mac = value

    @property
    def ip(self):
        """ the ip address of a target device """
        return self._ip

    @ip.setter
    def ip(self, value):
        """ set the ip address of a target device """
        self._ip = value

    @property
    def hostname(self):
        """
        the hostname of a target device. may or may not exist depending
        on the vendor of the device
        """
        return self._hostname

    @hostname.setter
    def hostname(self, value):
        """
        set the hostname of a target device. we can sometimes parse this out
        of the dhcp log messages
        """
        self._hostname = value

    @property
    def bundles(self):
        """ a list of all bundles associated with this target device """
        return self._bundles

    @bundles.setter
    def bundles(self, value):
        """ set the list of all bundles associated with this target device """
        self._bundles = value

    @property
    def vendor(self):
        """ get the vendor name of the target device """
        return self._vendor

    @vendor.setter
    def vendor(self, value):
        """ set the vendor name of the target device """
        self._vendor = value

    @property
    def files_downloaded(self):
        """ the list of all files downloaded by the target device """
        return self._files_downloaded

    @files_downloaded.setter
    def files_downloaded(self, value):
        """ replace the list of files downloaded with a new list """
        self._files_downloaded = value

    def to_json(self):
        """ serializes the TargetDevice to a json blob """
        return json.dumps({
            'mac': self.mac,
            'ip': self.ip,
            'hostname': self.hostname,
            'bundles': self.bundles,
            'vendor': self.vendor,
            'files_downloaded': self.files_downloaded,
            'provisioned': self.provisioned,
            'post_provision_scripts_run': self.post_provision_scripts_run,
        })

    def from_json(self, j):
        """ builds a TargetDevice from a json blob """
        obj = None
        try:
            obj = json.loads(j)
        except:
            raise LoadError("invalid json provided to from_json") from None

        new_t = TargetDevice(self.handle)
        if obj:
            if 'mac' in obj:
                new_t.mac = obj['mac']
            if 'ip' in obj:
                new_t.ip = obj['ip']
            if 'hostname' in obj:
                new_t.hostname = obj['hostname']
            if 'bundles' in obj:
                new_t.bundles = obj['bundles']
            if 'vendor' in obj:
                new_t.vendor = obj['vendor']
            if 'files_downloaded' in obj:
                new_t.files_downloaded = obj['files_downloaded']
            if 'provisioned' in obj:
                new_t.provisioned = obj['provisioned']
            if 'post_provision_scripts_run' in obj:
                new_t.post_provision_scripts_run = obj['post_provision_scripts_run']
        return new_t

    def save(self):
        """
        Store the TargetDevice in the Redis DB.
        Store a device:mac and device:ip key, for fast retrieval by ip or mac.
        """
        if self.ip is not None:
            self.handle.set(_key(self.ip), self.to_json())
        if self.mac is not None:
            self.handle.set(_key(self.mac), self.to_json())

    def add_bundle(self, bundle_name):
        """
        Add a bundle name to the list of bundles associated with the TargetDevice.
        Duplicate bundles are not added to the list.
        """
        if not bundle_name in self.bundles:
            self.bundles.append(bundle_name)

    def add_file_downloaded(self, file_name):
        """
        Add a file name to the list of files downloaded by the TargetDevice.
        Duplicate files are not added to the list.
        """
        if not file_name in self.files_downloaded:
            self.files_downloaded.append(file_name)

    def add_post_provision_script_run(self, file_name):
        """
        Add a file name to the list of post-provisioning scripts run for the TargetDevice.
        Duplicate files are not added to the list.
        """
        if not file_name in self.post_provision_scripts_run:
            self.post_provision_scripts_run.append(file_name)
