import json
from json import decoder


def escape(data):
    """
    escape: Escapes generic data as a string, returning either `"f{data}"` or
    `json.dumps("f{data}", ensure_ascii=True)`, where the former may be
    distinguished by lack of any leading double quote, and the latter will be
    used as necessary to ensure that all non-ascii characters are escaped and
    quoted (using double quotes), and that all whitespace characters and equal
    signs are escaped and/or quoted (space and equals chars just get quoted).

    There are two distinct cases (both only only ascii and non-zero length):
        1. raw python-formatted string, no whitespace, no equals, no leading
           double quote
        2. python-formatted string, encoded as json

    :param data:
    :return: str
    """
    data = f"{data}"
    escaped = json.dumps(data, ensure_ascii=True)
    if len(data) != 0 and data == escaped[1:len(escaped) - 1] and ' ' not in data and '=' not in data:
        return data
    return escaped


def encode(*args, **kwargs):
    """
    encode: Builds a single log line using the escape function from this
    package and guarantees no non-space whitespace in the result, note that
    means no trailing newline will be appended.

    :param args:
    :param kwargs:
    :return: str
    """
    return ' '.join([f"{escape(k)}={escape(kwargs[k])}" for k in kwargs] + [escape(arg) for arg in args])  # pylint: disable=consider-using-dict-items


def decode(data):
    """
    decode: Performs decoding of data encoded using the corresponding encode
    function from this package, the primary use case being parsing a single
    log line.

    :param data: str
    :return: Tuple<List<str>, Dict<str, str>>
    """
    if not isinstance(data, str):
        raise ValueError(f"logutil.decode: unexpected data ({type(data)}): {data}")
    args = []
    kwargs = {}
    d = json.JSONDecoder(strict=True)

    def read(i):
        i = decoder.WHITESPACE.match(data, i).end()
        if i >= len(data):
            return None, len(data)
        if data[i] != '"':
            s = i
            while i < len(data):
                if data[i] == '=' or data[i].isspace():
                    break
                i += 1
            return data[s:i], i
        field, i = d.raw_decode(data, i)
        if not isinstance(field, str):
            raise ValueError(f"logutil.decode: unexpected field ({type(field)}): {field}")
        return field, i

    i = 0
    while True:
        field, i = read(i)
        if field is None:
            return args, kwargs
        if i >= len(data) or data[i] != '=':
            args.append(field)
            continue
        kwargs[field], i = read(i + 1)
        if kwargs[field] is None:
            raise ValueError('logutil.decode: unexpected eof')
