from typing import Dict, Set, Optional
from central_sdi import storage_api
from central_sdi.api import schema

_key = 'policies'
_schema = schema.PolicyList(
    # improve compatibility with data older than the code
    # (e.g. when adding new required fields)
    partial=True,
)


def store(new_policies: Dict):
    """
    Update policies
    """
    storage_api.sa_store(_key, _schema.dumps(new_policies))


def load() -> Dict:
    """
    Load and return all policies, note that the backing data will be validated
    against the schema, excepting missing fields (potentially a gotcha in the
    migration case)
    """
    try:
        return _schema.loads(storage_api.sa_load(_key).read())
    except storage_api.ObjectNotFound:
        return _schema.load({"policies": []})


def flatten(policies: Dict) -> Optional[Dict[str, Set[str]]]:
    """
    Given policies (e.g. return value of load) flatten them down to map the
    set of zones allowed for each group, note that it will return None if the
    "policies" key did not exist.
    """
    policies = policies.get('policies')
    if policies is None:
        return None
    result = {}
    for policy in policies:
        for group in policy.get('groups', []):
            for zone in policy.get('zones', []):
                if group not in result:
                    result[group] = set()
                result[group].add(zone)
    return result
