import logging
import os

SERIAL_PORTS_FILE = "/var/run/serial-ports"

class OGCS():
    """
    This class encapsulates OGCS-specific functions for port discovery.
    """
    def get_all_ports(self):
        """
        Return the entire range of ports as a list, e.g. [1, 2, 3, 4]
        """
        num_ports = 0
        with open(SERIAL_PORTS_FILE, "r") as ports_file:
            for _ in ports_file:
                num_ports += 1

        return list(range(1, num_ports + 1))

    def get_available_ports(self, ports, skip_existing=False):
        """
        Take a list of ports, and return a subset of available ports.
        """
        port_config = os.popen('config -g config.ports').read()
        def _port_available(port):
            # Make sure port is enabled
            if f"config.ports.port{port}.mode portmanager" not in port_config:
                return False

            # Make sure there are no existing user sessions for this port
            if skip_existing and os.system(f"pmusers -n {port} | grep -q {port}") == 0:
                logging.getLogger().debug("Port {} has existing user session(s), skipping".format(port))
                return False
            return True

        available_ports = [port for port in ports if _port_available(port)]
        return available_ports

    def _get_config_element_for_port(self, config, port, element, get_backup=False):
        config_entry = f"config.ports.port{port}.{element}"
        if get_backup:
            config_entry = f"config.ports.port{port}.bak.{element}"

        for line in config:
            if config_entry in line:
                split = line.split(" ")
                if len(split) > 1:
                    return split[1]
        return None

    def apply_port_config(self):
        """
        Apply any config changes made.
        """
        if os.system("config -r serialconfig &> /dev/null"):
            logging.getLogger().error("Serial configurator failed")
            return False
        return True

    def set_port_config(self, port, baud=None, pinout=None, label=None, discovery_username=None, discovery_password=None):
        """
        Take a port, and configure it to the specified baud rate, pinout and label. If any
        of the parameters are not specified, or the existing config matches the parameter, that config
        will not be changed.

        Return a tuple of:
            - True if the operation succeeded, else False
            - True if the configuration is now dirty (i.e changes were made)
        """
        port_config = os.popen('config -g config.ports').read().split("\n")
        dirty = False

        if pinout:
            current_pinout = self._get_config_element_for_port(port_config, port, "pinout")
            if current_pinout != pinout:
                if os.system(f"config -s config.ports.port{port}.pinout={pinout}") != 0:
                    logging.getLogger().error("Failed to set port pinout")
                    return False, dirty
                dirty = True

        if baud:
            current_baud = self._get_config_element_for_port(port_config, port, "speed")
            if current_baud != baud:
                if os.system(f"config -s config.ports.port{port}.speed={baud}") != 0:
                    logging.getLogger().error("Failed to set port baud")
                    return False, dirty
                dirty = True

        if label:
            current_label = self._get_config_element_for_port(port_config, port, "label")
            if current_label != label:
                if os.system(f"config -s config.ports.port{port}.label={label}") != 0:
                    logging.getLogger().error("Failed to set port label")
                    return False, dirty
                dirty = True

        if discovery_username:
            current_username = self._get_config_element_for_port(port_config, port, "discovered.username")
            if current_username != discovery_username:
                if os.system(f"config -s config.ports.port{port}.discovered.username={discovery_username}") != 0:
                    logging.getLogger().error("Failed to set port discovered username")
                    return False, dirty
                dirty = True

        if discovery_password:
            current_password = self._get_config_element_for_port(port_config, port, "discovered.password")
            # Obfuscate password first
            obfuscated_password = os.popen(f"obfusc -o {discovery_password}").read()

            if current_password != obfuscated_password:
                if os.system(f"config -s config.ports.port{port}.discovered.password={obfuscated_password}") != 0:
                    logging.getLogger().error("Failed to set port discovered password")
                    return False, dirty
                dirty = True

        return True, dirty

    def backup_port_config(self, port):
        """
        Take a port, and backs up its current configuration so it can be restored later.

        For OGCS, this is done in-place e.g. "config.ports.port1.bak.label"

        Return True on success, or False on failure to set config
        """
        port_config = os.popen('config -g config.ports').read().split("\n")

        current_pinout = self._get_config_element_for_port(port_config, port, "pinout")
        current_baud = self._get_config_element_for_port(port_config, port, "speed")
        current_label = self._get_config_element_for_port(port_config, port, "label")

        config_cmd = ""
        if current_pinout:
            config_cmd = config_cmd + f"config -s config.ports.port{port}.bak.pinout={current_pinout} "
        if current_baud:
            config_cmd = config_cmd + f"-s config.ports.port{port}.bak.speed={current_baud} "
        if current_label:
            config_cmd = config_cmd + f"-s config.ports.port{port}.bak.label={current_label}"

        if os.system(config_cmd) != 0:
            logging.getLogger().error("Failed to backup port config")
            return False

        return True

    def restore_port_config(self, port):
        """
        Take a port, and restore any backed up configuration it has.

        Return True on success, or False on failure to set config
        """
        port_config = os.popen('config -g config.ports').read().split("\n")

        backup_pinout = self._get_config_element_for_port(port_config, port, "pinout", True)
        backup_baud = self._get_config_element_for_port(port_config, port, "speed", True)
        backup_label = self._get_config_element_for_port(port_config, port, "label", True)

        if backup_pinout:
            if os.system(f"config -s config.ports.port{port}.pinout={backup_pinout}") != 0:
                logging.getLogger().error("Failed to restore port pinout")
                return False

        if backup_baud:
            if os.system(f"config -s config.ports.port{port}.speed={backup_baud}") != 0:
                logging.getLogger().error("Failed to restore port baud")
                return False

        if backup_label:
            if os.system(f"config -s config.ports.port{port}.label={backup_label}") != 0:
                logging.getLogger().error("Failed to restore port label")
                return False

        return True
