import grpc
from typing import (
    Dict,
    Optional,
)
from netops import auth


def load_session(token: str) -> Optional[Dict[str, str]]:
    """
    load_session loads the key value pairs from the session file
    """
    try:
        r = {}
        with open(f"/sessions/session-{token}") as f:
            for l in f:
                l = l.rstrip().split('=', 2)
                if len(l) == 2:
                    r[l[0]] = l[1]
        return r
    except FileNotFoundError:
        return None


def grpc_channel() -> grpc.Channel:
    """
    grpc_channel returns a new channel, connecting to the nom-state socket
    (assumes lighthouse /var/run mounted as /var/run/host)
    """
    return grpc.insecure_channel('unix:///var/run/host/nom-state.sock')


def dummy_groups(channel: grpc.Channel, token: str) -> Dict:
    """
    dummy_groups allows us to avoid replacing the module implementations
    _immediately_, by returning an equivalent groups api response, based
    on what the session token has access to
    """
    stub = auth.AdapterStub(channel)
    principle = auth.Principle(session=auth.Session(token=token))
    resource = auth.Resource(lighthouse=auth.Lighthouse())
    action_ro = auth.Action(rights=auth.Rights(values=[auth.Rights.Value(entity='netops_modules')]))
    action_rw = auth.Action(rights=auth.Rights(values=[auth.Rights.Value(entity='netops_modules', write=True)]))
    if not stub.Can(auth.Can.Request(principle=principle, resource=resource, action=action_ro)).allowed:
        # pretend NodeUser
        return {
            "groups": [
                {
                    "enabled": True,
                    "mode": "smart_group",
                    "smart_group_roles": "NodeUser"
                }
            ]
        }
    if not stub.Can(auth.Can.Request(principle=principle, resource=resource, action=action_rw)).allowed:
        # pretend NodeAdmin
        return {
            "groups": [
                {
                    "enabled": True,
                    "mode": "smart_group",
                    "smart_group_roles": "NodeAdmin"
                }
            ]
        }
    # pretend LighthouseAdmin
    return {
        "groups": [
            {
                "enabled": True,
                "mode": "global",
                "global_roles": "LighthouseAdmin"
            }
        ]
    }
