"""
interface implements interface and connection selection logic for the purposes
of serving the ztp service (comprised of multiple different services including
dhcp and sftp)
"""

from copy import deepcopy
from typing import Callable, List

from netops.ngcs import models

class UndecidableException(BaseException):
    """
    UndecidableException is used to indicate no valid decision is available
    """


class Decider:
    """
    Decider models the actual decision making logic, and is set up to receive
    relevant information via lambda functions provided via it's constructor.

    Decision information is made available via properties, which are generated
    by calling the decide method.

    NOTE this is implementation could support manual (complete or partial)
         decision overrides, via implementation of setters (for example)

    UndecidableException will be thrown when making a decision, in the event
    that a decision cannot be reached (e.g. there is not enough data, or there
    is no valid interface with a valid connection).

    UndecidableException may also be thrown by any of the lambdas, to indicate
    that the current branch / case should be skipped / isn't viable. If no viable
    option is found, then these errors will be propagated / combined.
    """

    def __init__(
            self,
            system_interfaces: Callable[[], List[str]],
            system_mac_address: Callable[[str], str],
            ethtool_stdout: Callable[[str], str],
            api_firewall_zone: Callable[[str], models.FirewallZone],
            api_interface: Callable[[str], models.Interface],
            api_interface_by_id: Callable[[str], models.Interface],
            interface_link: Callable[[str], str],
            api_connections: Callable[[str], List[models.Connection]],
            system_failover_cellular: Callable[[], models.FailoverToCellular],
    ):
        """
        Constructs a new decider, note all lambdas with arguments receive the interface
        name (device in the api)

        :param system_interfaces: returns the interface names of all system interfaces
        :param system_mac_address: returns the mac address of an interface
        :param ethtool_stdout: returns the ethtool output for a single interface (like `ethtool <INTERFACE>`)
        :param api_firewall_zone: returns the api model for the zone of an interface
        :param api_interface: returns the api model for an interface, or None if it does not exist in the api (e.g. not exposed)
        :param api_interface_by_id: returns the api model for an interface specified by id, or None if it does not exist in the api (e.g. not exposed)
        :param interface_link: returns the link/peer of an interface
        :param api_connections: returns the connections for an interface
        """
        self._system_interfaces = system_interfaces
        self._system_mac_address = system_mac_address
        self._system_failover_cellular = system_failover_cellular
        self._ethtool_stdout = ethtool_stdout
        self._api_firewall_zone = api_firewall_zone
        self._api_interface = api_interface
        self._api_interface_by_id = api_interface_by_id
        self._interface_link = interface_link
        self._api_connections = api_connections
        self._mac_address = None
        self._interface = None
        self._connection = None

    @property
    def mac_address(self) -> str:
        """
        mac_address is the mac address of the decision interface as a string
        """
        return deepcopy(self._mac_address)

    @property
    def interface(self) -> models.Interface:
        """
        interface is the decision interface data from the api
        """
        return deepcopy(self._interface)

    @property
    def connection(self) -> models.Connection:
        """
        connection is the decision connection data from the api (of the interface)
        """
        return deepcopy(self._connection)

    def decide(self):
        """
        decide will (re)evaluate the secure provisioning interface decision
        """
        system_interfaces = []
        system_mac_address_map = {}
        api_interface_map = {}
        api_firewall_zone_map = {}
        api_connections_map = {}
        switch_interfaces = set()
        err_list = []

        # filter and map the data
        for system_interface in self._system_interfaces():
            try:
                # if failover settings is enabled, ignore that interface
                if self._system_failover_cellular().enabled and system_interface == self._system_failover_cellular().interface:
                    raise UndecidableException('Failover to Cellular is enabled in interface {}'.format(system_interface))

                # map interface data, might raise UndecidableException, which is fine (unless they all do)
                system_mac_address_map[system_interface] = self._system_mac_address(system_interface)
                api_interface_map[system_interface] = self._api_interface(system_interface)
                api_firewall_zone_map[system_interface] = self._api_firewall_zone(system_interface)
                api_connections_map[system_interface] = []

                # perform filtering logic on the interface level
                if not api_interface_map[system_interface].enabled:
                    raise UndecidableException('interface {} is not enabled'.format(system_interface))
                if api_firewall_zone_map[system_interface].name != 'lan':
                    raise UndecidableException('interface {} is not in the lan zone'.format(system_interface))

                # filter and map connection data
                for api_connection in self._api_connections(system_interface):
                    if api_connection.mode != 'static':
                        continue
                    settings = api_connection.ipv4_static_settings
                    if not settings:
                        continue
                    try:
                        _ = settings.netmask_cidr
                    except ValueError:
                        continue
                    api_connections_map[system_interface].append(api_connection)

                # extract ethtool metadata
                ethtool_stdout = self._ethtool_stdout(system_interface)
                if '1000baseKX/Full' in ethtool_stdout:
                    switch_interfaces.add(system_interface)
                elif api_interface_map[system_interface].slaves:
                    # If this is an aggregate interface, iterate over its slaves
                    # and check whether they are ALL ports on a switch interface. If so,
                    # consider this aggregate a switch.
                    switch_aggregate = True
                    for slave_id in api_interface_map[system_interface].slaves:
                        interface = self._api_interface_by_id(slave_id)
                        # Get linked interface
                        link = self._interface_link(interface.device)
                        if not link:
                            switch_aggregate = False
                            break
                        ethtool_stdout = self._ethtool_stdout(link)
                        if not '1000baseKX/Full' in ethtool_stdout:
                            switch_aggregate = False
                            break
                    if switch_aggregate:
                        switch_interfaces.add(system_interface)

                system_interfaces.append(system_interface)
            except UndecidableException as err:
                err_list.append(str(err))
        if not system_interfaces:
            raise UndecidableException(' | '.join(['central_dop.interface.Decider no valid decision'] + err_list))

        # we have at least one valid option, pick the best one
        # NOTE uses stable sorts, the lower priority sort criteria are applied first

        # string sort (reversed)
        system_interfaces.sort(reverse=True)

        # put wwan* last
        system_interfaces.sort(key=lambda v: v[:4] == 'wwan')

        # put net* first
        system_interfaces.sort(key=lambda v: v[:3] != 'net')

        # put switch interfaces first
        system_interfaces.sort(key=lambda v: v not in switch_interfaces)

        # prefer name *-nom-*
        api_connections_map[system_interfaces[0]].sort(key=lambda v: '-nom-' not in v.name)

        # prefer enabled
        api_connections_map[system_interfaces[0]].sort(key=lambda v: not v.enabled)

        # write out the new decision
        self._mac_address = system_mac_address_map[system_interfaces[0]]
        self._interface = api_interface_map[system_interfaces[0]]
        self._connection = api_connections_map[system_interfaces[0]][0] if api_connections_map[system_interfaces[0]] else None
