#!/usr/bin/env bash

# performs runtime configuration then execs supervisord

set -o pipefail

die() { ( >&2 echo FATAL "$@" ); exit 1; }

# default environment variables
if [ -z "$LH_HOST_ADDRESS" ]; then
    LH_HOST_ADDRESS="$( ( IFS=" " && parts=( $( ip r show default ) ) && [ ! -z "${parts[2]}" ] && echo "${parts[2]}" ) )" \
        || die "LH_HOST_ADDRESS: failed to identify host address"
fi
if [ -z "$LH_SSH_USERNAME" ]; then
    LH_SSH_USERNAME="root"
fi
if [ -z "$LH_API_VERSION" ]; then
    LH_API_VERSION="v3.2"
fi
if [ -z "$LH_DOCKER_BRIDGE" ]; then
    LH_DOCKER_BRIDGE="$( ssh "$LH_SSH_USERNAME"@"$LH_HOST_ADDRESS" 'docker network inspect netops' | jq -r '.[0].Id' )" \
        && LH_DOCKER_BRIDGE="${LH_DOCKER_BRIDGE:0:12}" \
        && [ ! -z "$LH_DOCKER_BRIDGE" ] \
        && LH_DOCKER_BRIDGE="br-$LH_DOCKER_BRIDGE" \
        || die "LH_DOCKER_BRIDGE: failed to identify docker bridge"
fi

# taking the system version via environment variable will not pick up the new lighthouse version unless the deployment
# container is recreated (e.g. via sync)
LH_SYSTEM_VERSION="$( ssh "$LH_SSH_USERNAME"@"$LH_HOST_ADDRESS" 'cat /etc/sw_version_internal' )" \
    && [ ! -z "$LH_SYSTEM_VERSION" ] \
    || die "LH_SYSTEM_VERSION: failed to identify system version"

# export env variables
export LH_HOST_ADDRESS
export LH_SSH_USERNAME
export LH_API_VERSION
export LH_DOCKER_BRIDGE
export LH_SYSTEM_VERSION

# validate env variables
lhDockerBridge="$( printf '%q' "$LH_DOCKER_BRIDGE" )" \
    && [ ! -z "$lhDockerBridge" ] \
    && ssh "$LH_SSH_USERNAME"@"$LH_HOST_ADDRESS" "ip a show ${lhDockerBridge} > /dev/null" \
    || die "LH_DOCKER_BRIDGE|LH_HOST_ADDRESS|LH_SSH_USERNAME: failed to validate docker bridge over ssh"

# firewall rules required to correctly isolate netops services
{ configure-firewall-post deployment "$LH_SSH_USERNAME"@"$LH_HOST_ADDRESS" << EOF
# marks - https://www.tldp.org/HOWTO/Adv-Routing-HOWTO/lartc.netfilter.html
# 2147483648: whitelists traffic that would otherwise be blocked by filter:NETOPS-DOCKER
#
# allows host port 2222 to be forwarded to central-dop
iptables -I PREROUTING -t mangle -p tcp --dport 2222 -m addrtype --dst-type LOCAL -j MARK --set-mark 2147483648

# filter:NETOPS-DOCKER
#
# block unwanted traffic from being forwarded to the netops docker bridge network
iptables -N NETOPS-DOCKER
iptables -A NETOPS-DOCKER -i net+ -o ${lhDockerBridge} -m mark --mark 2147483648 -j RETURN
iptables -A NETOPS-DOCKER -i net+ -o ${lhDockerBridge} -m conntrack --ctstate RELATED,ESTABLISHED -j RETURN
iptables -A NETOPS-DOCKER -i net+ -o ${lhDockerBridge} -j Block
iptables -A NETOPS-DOCKER -i net+ -o ${lhDockerBridge} -j DROP
iptables -A NETOPS-DOCKER -j RETURN
#
iptables -I DOCKER-USER -j NETOPS-DOCKER
EOF
} \
    || die "firewall-post: failed to configure lighthouse firewall"

exec -- "$@"
