"""
prov_manager provides a class that manages the running provisioning scripts as
spawned by provd. The maanger will run the scripts, track their status, and cancel
the process after it has run for the duration of the timeout.
"""
import subprocess
import time
import os
import threading
import sys
import signal
import psutil

from .log import Log


def sprint(msg):
    """print to stdout"""
    print("[provd] {}".format(msg), flush=True)

def eprint(msg):
    """print to stderr"""
    print("[provd] {}".format(msg), file=sys.stderr, flush=True)


class ProvManager:
    """
    ProvManager keeps track of processes currently running for provd.
    """

    def __init__(self):
        self._running_count = 0
        self._procs = []
        self._thread = threading.Thread(target=self.check_timeouts)
        self._thread_stopped = False
        self._count_lock = threading.Lock()
        self._thread_lock = threading.Lock()

    @property
    def procs(self):
        """ a list of all running processes """
        return self._procs

    @property
    def running_count(self):
        """ a count for the number of processes. should match len(self.procs) """
        with self._count_lock:
            return self._running_count

    @running_count.setter
    def running_count(self, value):
        """ set the count for the number of processes """
        with self._count_lock:
            self._running_count = value

    @property
    def thread_stopped(self):
        """
        returns true when the main process stops the thread that prov manager
        runs in
        """
        with self._thread_lock:
            return self._thread_stopped

    @thread_stopped.setter
    def thread_stopped(self, value):
        """
        set the value of thread_stopped to indicate processes should be closed
        as it is used as a sentinel value in check_timeouts
        """
        with self._thread_lock:
            self._thread_stopped = value

    def add_proc(self, process):
        """ add another process (i.e. script) to manage """
        self.procs.append(process)
        self.running_count = self.running_count + 1
        return True

    def remove_proc(self, process):
        """ remove a process from the currently running list """
        self.procs.remove(process)
        self.running_count = self.running_count - 1

    def check_timeouts(self):
        """
        check_timeouts ensures that processes running past their timeout period
        are terminated, and cleans up scripts which exited normally.
        """
        while not self.thread_stopped:
            for proc in self.procs:
                if proc.terminated():
                    sprint("completed post-provisioning script {} for host {} with exitcode {}".format(
                        proc.actual_script, proc.mac, proc.returncode))
                    self.remove_proc(proc)
                    proc.cleanup()
                else:
                    if proc.expired():
                        eprint("killing post-provisioning script {} for host {} for running too long".format(
                            proc.actual_script, proc.mac))
                        proc.term()
                        self.remove_proc(proc)
            time.sleep(1)  # don't be too busy

    def start(self):
        """
        start begins the thread which will run in the background monitoring
        for scripts that have timed out.
        """
        self._thread.start()

    def stop(self):
        """ stop the thread monitoring the processes """
        self.thread_stopped = True

def monitor_stdout(proc, mac, script):
    """
    This function reads from a stdout PIPE, and logs each line. Once the process terminates,
    the iterator will exit.
    """
    logger = Log('post_provision_script_stdout')
    logger.mac = mac
    logger.script = script

    for line in proc.stdout:
        logger.script_stdout = line.decode("utf-8", errors="replace")
        logger.print_log()

class Proc:
    """
    Proc runs a subprocess, as well as a thread to logs the stdout + stderr of the process.
    """
    def __init__(self, tmp_script, actual_script, timeout, mac):
        self._timeout = timeout
        self._mac = mac
        self._tmp_script = tmp_script
        self._actual_script = actual_script
        self._process = subprocess.Popen([self._tmp_script], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)  # pylint: disable=consider-using-with
        self._monitor_thread = threading.Thread(
            target=monitor_stdout, args=[self._process, self._mac, self._actual_script], daemon=True)
        self._monitor_thread.start()
        self._start_time = time.time()
        self._state = None

    def cleanup(self):
        """
        cleanup removes the temporary script file from disk, and closes the
        thread used by the monitor if it is still hanging around
        """
        if os.path.exists(self._tmp_script):
            os.remove(self._tmp_script)

        # Make sure monitor thread has finished
        if self._monitor_thread.is_alive():
            try:
                self._monitor_thread.join(timeout=5)
            except RuntimeError as err:
                eprint("Failed to cleanup monitor thread: {}".format(err))

    @property
    def timeout(self):
        """ the timeout for the script to be terminated """
        return self._timeout

    @property
    def mac(self):
        """ the mac address to track which device triggered a script run """
        return self._mac

    @property
    def tmp_script(self):
        """ the location of the script after it has been templated """
        return self._tmp_script

    @property
    def actual_script(self):
        """ the location of the pre-templated script """
        return self._actual_script

    @property
    def process(self):
        """ the Subprocess object for the current Proc """
        return self._process

    @property
    def state(self):
        """ indicates the state of the process running """
        return self._state

    @property
    def returncode(self):
        """ the exit status of the script that has just been run """
        return self._process.returncode

    @state.setter
    def state(self, s):
        """ just set the state of the proc to the given value """
        self._state = s

    def terminated(self):
        """
        Check whether process is terminated. The poll function sets the returncode attribute on process.
        """
        if self.returncode is not None:
            return True
        return self._process.poll() is not None

    def expired(self):
        """
        true if the Proc has run for longer than the timeout value
        """
        return (time.time() - self._start_time) > self.timeout

    def term(self):
        """
        Terminate the Proc, including all child processes that may have been
        spawned.
        """
        try:
            parent = psutil.Process(self.process.pid)

            # Kill all children spawned in subprocesses.
            children = parent.children(recursive=True)
            for process in children:
                process.send_signal(signal.SIGKILL)
            parent.send_signal(signal.SIGKILL)
            self.cleanup()

        except psutil.NoSuchProcess:
            pass
