"""
Provides functions to determine whether prerequisites for
ordered deployment of bundles have been satisfied

For every bundle, the user can specify other bundles that are required to be provisioned first
This module checks whether or not these specified bundles have been provisioned to devices (i.e. all required files served)
"""
from .target_device import TargetDevice

def dependencies_satisfied(redis_handle, target_device, filename, config):
    """
    Determines if the dependencies required for @target_device
    to be provisioned have been satisfied
    """

    # cumulus boot loader scripts exempted due to the way cumulus implements ZTP
    if filename.endswith("_boot.cumulus.sh.j2"):
        return True

    dependencies = []

    # Find the dependencies for the target device
    for resource in config.resources:
        # find the correct resource bundle
        if resource.name in target_device.bundles:
            dependencies = resource.provisionAfter

    # mac addresses of all devices that have been provisioned already
    provisioned_devices = []

    # look at every device that the OM has seen
    for key in redis_handle.scan_iter(match="device*"):
        other_device = TargetDevice(redis_handle).from_json(redis_handle.get(key))

        # only look at each device once and ignore devices that haven't been provisioned yet
        if other_device.provisioned and other_device.mac not in provisioned_devices:
            provisioned_devices.append(other_device.mac)

            for bundle in other_device.bundles:
                if bundle in dependencies:
                    dependencies.remove(bundle)

    return len(dependencies) == 0
