from typing import (
    Iterable,
    Dict,
    Any,
)


class NodeNetOpsModule:
    """
    Represents a node's NetOps module property
    """

    def __init__(self, module_data):
        self._name = module_data.get("name")
        self._version = module_data.get("version")

    @property
    def name(self):
        return self._name

    @property
    def version(self):
        return self._version


class Node:
    """
    Represents a Lighthouse node object as retrieved by the LH REST API, with fields that we care about
    """

    # TODO split this into separate params
    def __init__(self, node_data: Dict[str, Any]):
        self._id = node_data.get("id")
        self._lhvpn_address = node_data.get("lhvpn_address")
        self._mac_address = node_data.get("mac_address")
        self._model = node_data.get("model")
        self._name = node_data.get("name")
        self._runtime_status = node_data.get("runtime_status")
        self._og = self._model and (self._model.startswith('ACM') or self._model.startswith('IM') or self._model.startswith('VACM'))
        self._modules = [NodeNetOpsModule(module) for module in node_data.get("nom_modules", [])]
        self._rights = node_data.get("rights")

    @property
    def id(self):
        """ returns the Lighthouse config ID of the node """
        return self._id

    @property
    def lhvpn_address(self):
        """ returns the LHVPN address of the node """
        return self._lhvpn_address

    @property
    def mac_address(self):
        """ returns the MAC address of the node"""
        return self._mac_address

    @property
    def model(self):
        """ returns the model of the node """
        return self._model

    @property
    def name(self):
        return self._name

    @property
    def connection_status(self):
        """ return connection status of the node # TODO do this better """
        return self._runtime_status.get('connection_status')

    @property
    def modules(self) -> Iterable[NodeNetOpsModule]:
        """ returns the node's NetOps modules """
        return self._modules

    @property
    def og(self):
        """ Returns True if the node is an OG console server (VACM, ACM700x or IM72xx) """
        return self._og

    @property
    def rights(self) -> Dict[str, bool]:
        return dict(self._rights) if self._rights is not None else {}

    def get_module_by_name(self, name):
        for module in self.modules:
            if module.name == name:
                return module

        return None
