"""
inventory is a module which provides functions to generate an Ansible dynamic inventory based on a node_inventory
list.
"""
import sys
import os
import central_dop.config
from central_dop import nodes
from netops.lighthouse import nodes as lh_nodes


def add_node_to_inventory(node_id, all_nodes, all_og_nodes, meta, config):
    """
    Retrieves details for a node based on node ID, and adds it to the Ansible inventory
    """
    try:
        # TODO if the dop codebase sees a significant refactor, strongly consider
        #      changing this to avoid the direct use of an environment variable this
        #      far into the source
        # WARNING should be None in the case of deploy via the git workflow
        token = os.getenv("USER_TOKEN")
        node = lh_nodes.Node(nodes.make_request(f"https://{nodes.get_host_address()}/api/v3/nodes/{node_id}", token=token)["node"])
        if token and not node.rights.get('netops', False):
            # NOTE rights are broken for cert auth, and cert auth is necessary for the git workflow
            raise central_dop.nodes.Error('not authorized')
        if not node.lhvpn_address:
            raise central_dop.nodes.Error('missing lhvpn address')
        if not node.model:
            raise central_dop.nodes.Error('missing model')

        # Validate the node has Secure Provisioning activated
        dop_module = [module for module in node.modules if module.name == "dop"]
        if not dop_module:
            return

        if node.og:
            all_og_nodes.update([node.lhvpn_address])
        else:
            all_nodes.update([node.lhvpn_address])

        node_inv = f"{node_id}".replace("-", "_")
        node_vars = {}

        if config and 'node_vars' in config and config['node_vars'] and node_inv in config['node_vars'] and config['node_vars'][node_inv]:
            for k in config['node_vars'][node_inv]:
                node_vars[k] = config['node_vars'][node_inv][k]

        node_vars['identifier'] = node_inv

        meta['hostvars'][node.lhvpn_address] = node_vars
    except central_dop.nodes.Error as err:
        print(f"Error adding node {node_id} to inventory: {err.message}", file=sys.stderr)


def get_inventory(config):
    """
    Loads the YAML config file and produces a corresponding Ansible dynamic inventory
    """
    # Create a set to store all nodes that we are deploying to. This will be built up as each node inventory
    # is evaluated below. Making it a set avoids duplicate nodes in the list.
    all_nodes = set()
    all_og_nodes = set()

    # This is used to store identifier vars for each node (corresponds to the resource directory created by
    # the other scripts)
    meta = {"hostvars": {}}
    root_inventory = {
        "_meta": meta
    }

    deployment_list = config["node_inventory"]
    for node_inventory in list(config["node_inventory"].keys()):
        for kind in deployment_list[node_inventory]:
            if kind == "smartgroup":
                node_ids = []
                try:
                    node_ids = nodes.resolve_smartgroup_to_ids(deployment_list[node_inventory][kind])
                except central_dop.nodes.Error as err:
                    print("Error generating dynamic inventory from smartgroup {}: {}".format(
                        deployment_list[node_inventory][kind], err.message))

                for node_id in node_ids:
                    add_node_to_inventory(
                        node_id,
                        all_nodes,
                        all_og_nodes,
                        meta,
                        config,
                    )

            elif kind == "static":
                node_ids = deployment_list[node_inventory][kind]
                for node_id in node_ids:
                    add_node_to_inventory(
                        node_id,
                        all_nodes,
                        all_og_nodes,
                        meta,
                        config,
                    )

    root_inventory["remote"] = {
        "hosts": sorted(list(all_nodes)),
        "vars": {
            "ansible_port": 2222,
        },
    }

    root_inventory["remote_og"] = {
        "hosts": sorted(list(all_og_nodes)),
        "vars": {
        },
    }

    return root_inventory
