"""
config is a module for loading the configuration file from a given path, and various
configuration functions.
"""
import yaml
import re
import os
from glob import glob

from central_dop.prov_config import Config, Resource


def strip_extension(filename, ext=".j2"):
    """
    strip_extension: removes the .j2 extension from a filename if it has
    that extension.
    """
    if filename is None:
        return None
    if filename.endswith(ext):
        return os.path.splitext(filename)[0]
    return filename


class Error(Exception):
    """Base class for exceptions in the Config module."""
    def __init__(self, msg): # pylint: disable=super-init-not-called
        self._message = msg

    @property
    def message(self):
        """ return the exception's message """
        return self._message

class InputError(Error):
    """
    InputError is raised when the file provided is missing required configuration.
    """

class InventoryError(Error):
    """
    InventoryError is raised when a node inventory name contains invalid data.
    """

class ResourceError(Error):
    """
    ResourceError is raised when a resource bundle contains invalid data.
    """

def load_config_file():
    """
    load_config_file is deprecated but kept for backwards compatibility. Prefer
    the load_config_from_dir function instead.
    """
    config_dir = "/srv/deploy"
    return load_config_from_dir(config_dir)


def load_config_from_dir(directory):
    """
    load_config_from_dir finds the config.(yml|yaml) file in the given directory.
    If more than one file was found, the first (as returned by glob()) will be
    parsed and returned.
    If no files were found, None is returned.
    """
    if not os.path.isdir(directory):
        raise InputError("directory provided was not a valid path")

    # If there are multiple YAML files in the directory, choose the first
    # (ordered by name).
    config_file = ""
    for file in sorted(glob('{}/*.yml'.format(directory)) + glob('{}/*.yaml'.format(directory))):
        config_file = file
        break

    if config_file == "":
        raise InputError("directory does not contain a valid config file")

    with open(config_file, 'r') as f:
        config = yaml.safe_load(f)
        try:
            validate_config(config)
            return config
        except Error as err:
            raise ResourceError(str(err)) from None


def validate_config(config):
    """
    validate_config is a function that parses the given config and returns
    any relevant errors. The exceptions are specific to the type of error.
    """
    if not "device_resources" in config:
        raise InputError(
            "configuration file does not contain any resource groups")

    if not "deployment" in config:
        raise InputError(
            "configuration file does not contain any deployment information")

    # Node inventory and device resources names need to be validated before continuing.
    # The deployment uses the names for directories, so it needs to be unix
    # directory friendly.
    name_r = '^[a-zA-Z0-9_]+$'
    name_check = re.compile(name_r)

    # Validate that all specified resources for deployment are included in the
    # list of device resources.
    for name in config['deployment']:
        if not name_check.match(name):
            raise InventoryError(
                ("invalid name provided ({}), names must be alphanumeric and can only contain an underscore symbol".format(name)))
        for resource_name in config['deployment'].get(name):
            if resource_name not in config['device_resources']:
                raise ResourceError(
                    "resource ({}) does not exist in device_resources".format(resource_name))

    # Create a set of all resource filenames for validating naming conflicts
    resource_filenames = []

    # Validate the device resources.
    device_resources = config['device_resources']
    for name in device_resources:
        if not name_check.match(name):
            raise InventoryError(
                ("invalid name provided ({}), names must be alphanumeric and can only contain an underscore symbol".format(name)))

        device_resource = device_resources.get(name)

        if not device_resource.get('device_type'):
            raise ResourceError("resources must have a device type specified")

        if device_resource.get('device_type') == "cloud_provisioned" and device_resource.get('provision_after'):
            raise ResourceError("ordered provisioning is not supported for cloud provisioned devices")

        # We support 3 types of mac addresses to be specified in the yaml:
        # - a full mac address (6 octets)
        # - a partial (glob) mac address (1+ octet). We don't support a glob match
        #   for specific octets of a mac address.
        # - a reverse match of either of the first two (beginning with a !)
        if device_resource.get('mac_address'):
            # A mac is invalid if:
            # - it is greater than 6 octets in length,
            # - a glob pattern is not the right-most character,
            # - a glob pattern is the first character
            for mac in device_resource.get('mac_address'):
                glob_pos = mac.find("*")
                if glob_pos == 0 or (glob_pos == 1 and mac[0] == "!"):
                    raise ResourceError(
                        "a mac may not attempt to match a wildcard (*) on the first character")
                if glob_pos == -1:
                    if len(mac.split(":")) != 6:
                        raise ResourceError(
                            "a mac must be only 6 octets if there is no wildcard (*) match")
                    if len(mac.replace(":", "").replace("!", "")) != 12:
                        raise ResourceError(
                            "a mac must be only 6 octets if there is no wildcard (*) match")
                elif glob_pos > 0:
                    if glob_pos+1 != len(mac):
                        raise ResourceError(
                            "the wildcard (*) match can only be in the rightmost position of a mac")

        # Check that all listed dependency bundles for ordered provisioning are actual bundles
        if device_resource.get('provision_after'):
            for dependencyBundle in device_resource.get('provision_after'):
                if dependencyBundle not in device_resources:
                    raise ResourceError(
                        "{} is not a valid resource bundle".format(dependencyBundle)
                    )
                if device_resources[dependencyBundle].get('device_type') == "cloud_provisioned":
                    raise ResourceError(
                        "{} is invalid, {} does not support OrderedProvisioning".format(name, dependencyBundle)
                    )

        # We only support serial number filename substitution for CiscoXR devices.
        if device_resource.get('serial_number'):
            if device_resource.get('device_type') != 'cisco_xr':
                raise ResourceError(
                    "serial number substitution is only supported for cisco_xr devices")

        if device_resource.get("device_type") == "opengear":
            if device_resource.get("model"):
                validate_opengear_model(device_resource.get("model"))
            if device_resource.get("lighthouse"):
                validate_lighthouse_info(device_resource.get("lighthouse"))

        for resource in ["config_file", "image_file", "script_file"]:
            if device_resource.get(resource):
                resource_filenames.append(device_resource.get(resource))

    # Validate that user has not specified a templated and non-templated file with the same name
    # e.g. "arista.cfg" and "arista.cfg.j2".
    name_conflict = [file for file in resource_filenames if file.endswith('.j2') and file[:-3] in resource_filenames]
    if name_conflict:
        raise ResourceError("duplicate resource filenames [{}, {}]".format(name_conflict[0][:-3], name_conflict[0]))

    def cycle_util(resources, node, visited, path):
        visited[node] = True
        path.append(node)

        if "provision_after" in resources.get(node):
            for dependency in resources.get(node).get("provision_after"):
                if not visited[dependency]:
                    if cycle_util(resources, dependency, visited, path):
                        return True
                elif dependency in path:
                    return True

        path.pop()
        return False

    visited = {resource: False for resource in device_resources}
    for resource in device_resources:
        path = []
        if not visited[resource]:
            if cycle_util(device_resources, resource, visited, path):
                raise ResourceError("circular provisioning chain detected")

def validate_opengear_model(model):
    """
    validates the opengear model is in the list of supported models
    """
    supported = ['ACM700x', 'CM71xx', 'CM7196',
                 'ACM7004-5', 'IM72xx', 'OM22xx','OM12xx', 'CM81xx']
    if not model in supported:
        raise ResourceError(
            "invalid model '{}' for device type opengear".format(model))


def validate_lighthouse_info(lighthouse_data):
    """
    validates required lighthouse enrollment config is included
    we support url, bundle, token, and port. if url is included, at least token
    must be included.
    """
    if not lighthouse_data.get("url"):
        raise ResourceError(
            "url is required for Lighthouse enrollment configuration")
    if not lighthouse_data.get("token"):
        raise ResourceError(
            "token is required for Lighthouse enrollment configuration")


def generate_provd_config(config, group_name):
    """
    This produces a configuration file for provd based on the config provided.
    The config needs to include all device resources that are to be deployed
    to @group_name (stripped of j2 extensions to match the filenames actually
    requested by the devices).
    """
    out = Config()
    resource_list = config.get('deployment').get(group_name)
    for res_name in resource_list:
        dr = config.get('device_resources').get(res_name)
        if dr.get("post_provision_script"):
            script = "/scripts/{}".format(
                dr.get("post_provision_script"))
        else:
            script = ""
        r = Resource(res_name, script)
        if dr.get('config_file'):
            r.files.append(strip_extension(dr.get('config_file')))
        if dr.get('image_file'):
            r.files.append(strip_extension(dr.get('image_file')))
        if dr.get('script_file'):
            r.files.append(strip_extension(dr.get('script_file')))
        if dr.get('post_provision_script_timeout'):
            r.timeout = dr.get('post_provision_script_timeout')
        if dr.get('provision_after'):
            r.provisionAfter = dr.get('provision_after')
        out.add_resource(r)
    return out.to_json()
