import io
import os
import requests

from netops import util


class ObjectNotFound(Exception):
    """
    indicates that an object was not found at the provided storage API key
    """


def sa_address():
    """
    sa_address is the base address for the relevant storage api

    :return: str
    """
    address = os.getenv('STORAGE_API_ADDR')
    if not address:
        raise ValueError('STORAGE_API_ADDR not set')
    return address


def sa_store(key, value):
    """
    stores arbitrary data in the workspace api, note that value should be an
    iterable for streaming of large files

    :param key:
    :param value:
    """
    requests.post(f"{sa_address()}/{key}", data=value).raise_for_status()


def sa_load(key):
    """
    loads arbitrary data in the workspace api, streaming the response

    you might want to read this the docs re: streaming behavior
    https://2.python-requests.org/en/master/user/quickstart/#raw-response-content

    :param key:
    :return: stream
    """
    response = requests.get(f"{sa_address()}/{key}", stream=True)
    if response.status_code == 404:
        raise ObjectNotFound('workspace object not found: {}'.format(key))
    response.raise_for_status()
    return io.BufferedReader(util.IterableStream(response.iter_content()), buffer_size=io.DEFAULT_BUFFER_SIZE)


def sa_delete(key):
    """
    deletes arbitrary data from the workspace api, succeeding if it didn't exist
    """
    response = requests.delete(f"{sa_address()}/{key}")
    if response.status_code == 404:
        return
    response.raise_for_status()
