"""Module publishd implements a simple daemon that periodically publishes
user-specified device configuration to the ag-devices daemon"""

import abc
import asyncio
import contextlib
import logging
import grpc
from typing import (
    Coroutine,
    Callable,

)
from netops import (
    aioutil,
    automation_gateway as ag,
)
from datetime import (
    timedelta,
)


def _ticker_poll_device_names():
    return aioutil.ticker(timedelta(minutes=3), initial=True)


class Daemon:
    def __init__(
            self,
            *,
            logger: logging.Logger,
            ag_devices: ag.DevicesStub,
            device_names: 'DeviceNames',
    ):
        self._es = contextlib.AsyncExitStack()
        self._done = aioutil.Done()
        self._tasks = set()
        self._logger = logger
        self._ag_devices = ag_devices
        self._device_names = device_names

    @property
    def done(self) -> aioutil.Done:
        return self._done

    async def run(self):
        err = None
        async with self._es:
            await self._es.enter_async_context(self._wait())
            try:
                self._create_task(self._log_stopping())
                # the first exception in tasks or self.done will start shutdown
                tasks = {
                    self.done.future,
                    self._create_task(self._poll_device_names()),
                }
                while tasks and not self.done:
                    done, _ = await asyncio.wait(tasks, return_when=asyncio.FIRST_COMPLETED)
                    if done:
                        tasks.difference_update(done)
                        for task in done:
                            await task
            except Exception as e:  # pylint: disable=broad-except
                # propagate any exception (e.g. cancellation) only after we wait for all tasks
                err = e
                self._logger.error('publishd error (%s): %s', type(err).__name__, err)
            finally:
                # indicate that we are stopping prior to waiting for background tasks to finish
                self._done.close()
        if err is not None:
            raise err

    @contextlib.asynccontextmanager
    async def _wait(self):
        """implements a context manager that waits for any running tasks (that
         were added to self._tasks), just prior to exiting run"""
        yield None
        while self._tasks:
            done, _ = await asyncio.wait(self._tasks)
            if done:
                self._tasks.difference_update(done)
                for task in done:
                    try:
                        await task
                    except Exception as err:  # pylint: disable=broad-except
                        self._logger.warning('publishd.wait [%s] error (%s): %s', task, type(err).__name__, err)

    def _create_task(self, cr: Coroutine):
        """intended to be used to "call soon" an operation that must finish prior to exit"""
        task = asyncio.create_task(cr)
        self._tasks.add(task)
        return task

    async def _log_stopping(self):
        await self._done
        self._logger.info('publishd stopping')

    async def _poll_device_names(self):
        self._logger.info('publishd.poll_device_names started')
        try:
            async for _ in await self._es.enter_async_context(_ticker_poll_device_names()):
                if self._done:
                    return
                try:
                    self._logger.debug('publishd.poll_device_names load')
                    req = ag.DevicesPublish.Request(manual_mac_name=await asyncio.wait_for(self._done.wait(self._device_names.load()), 120))
                    self._logger.debug('publishd.poll_device_names publish: %s', _LazyStr(req, lambda v: v.replace('\n', '')))
                    await asyncio.wait_for(self._done.wait(self._ag_devices.Publish(req)), 60)
                    self._logger.debug('publishd.poll_device_names success')
                except aioutil.DoneError:
                    return
                except (RuntimeError, IOError, asyncio.TimeoutError, grpc.RpcError) as err:
                    self._logger.error('publishd.poll_device_names error (%s): %s', type(err).__name__, err)
        finally:
            self._logger.info('publishd.poll_device_names stopped')


class DeviceNames(abc.ABC):
    """models interactions with the device names config"""

    @abc.abstractmethod
    async def load(self) -> ag.Manual.MacName:
        raise NotImplementedError

    @classmethod
    def __subclasshook__(cls, c):
        if cls is DeviceNames:
            return _check_methods(
                c,
                "load",
            )
        return NotImplemented


def _check_methods(c, *methods):
    mro = c.__mro__
    for method in methods:
        for b in mro:
            if method in b.__dict__:
                if b.__dict__[method] is None:
                    return NotImplemented
                break
        else:
            return NotImplemented
    return True


class _LazyStr:
    def __init__(self, value, fn: Callable[[str], str]):
        self._value = value
        self._fn = fn

    def __str__(self):
        return self._fn(f"{self._value}")
