"""
implements /nodes/{node_id}/status
"""

import falcon
import grpc
from typing import (
    Optional,
    Callable,
    List,
    Dict,
)
from central_ag.api import (
    schema,
    util,
)
from netops import (
    automation_gateway as ag,
)
from netops.lighthouse import (
    nodes as lh_nodes,
)
from google.protobuf import (
    empty_pb2,
)


class Resource:
    schema = schema.NodeStatus()

    def __init__(
            self,
            *,
            get_nodes: Optional[Callable[[str, List[str]], Dict[str, lh_nodes.Node]]] = None,  # (token, node_ids)
    ):
        self._get_nodes = get_nodes

    def on_get(self, req: falcon.Request, res: falcon.Response, node_id: str):  # pylint: disable=unused-argument
        """
        ---
        description: Informs on the Automation Gateway status of an enrolled node.
        parameters:
            - name: node_id
              in: path
              required: true
              schema:
                type: string
        responses:
            200:
                description: OK
                content:
                    application/json:
                        schema: NodeStatus
            404:
                description: Not Found
        """
        nodes = self._get_nodes(req.context['token'], [node_id])
        node = nodes.get(node_id)
        if not node:
            raise falcon.HTTPNotFound

        # initialise conditions, see also netops.automation_gateway.Condition
        conditions = {}
        for condition in (
                {
                    'type': 'Ready',
                    'status': 'False',
                },
                {
                    'type': 'GuacReady',
                    'status': 'False',
                },
                {
                    'type': 'GuacActive',
                    'status': 'False',
                },
        ):
            conditions[condition['type']] = condition

        if not node.lhvpn_address:
            conditions['Ready']['reason'] = 'LhvpnAddress'
            conditions['Ready']['message'] = 'Missing LHVPN address'
        elif not node.mac_address:
            conditions['Ready']['reason'] = 'MacAddress'
            conditions['Ready']['message'] = 'Missing MAC address'
        elif not any(mod.name == 'ag' for mod in node.modules):
            conditions['Ready']['reason'] = 'Unactivated'
            conditions['Ready']['message'] = 'Not activated for Automation Gateway (ag)'
        elif not util.node_sufficient_rights(nodes, ag.Node.Reference(id=node_id, mac_address=node.mac_address, ip_address=node.lhvpn_address)):
            conditions['Ready']['status'] = 'Unknown'
            conditions['Ready']['reason'] = 'Forbidden'
            for condition_type in ('GuacReady', 'GuacActive'):
                conditions[condition_type]['status'] = 'Unknown'
        else:
            # TODO it'd be nice to ask the remote service how many guacamole sessions it supports
            max_guacamole_sessions = 1
            try:
                with util.ag_remote_channel(node.lhvpn_address) as ch:
                    guacamole_sessions: List[ag.GuacamoleSession] = list(ag.RemoteStub(ch).ListGuacamoleSessions(empty_pb2.Empty()).guacamole_sessions)
            except grpc.RpcError as err:
                err: grpc.Call
                conditions['Ready']['reason'] = f"Grpc{''.join(err.code().value[1].title().split())}"
                conditions['Ready']['message'] = f'Unexpected gRPC error: {err.details()}'
                conditions['GuacActive']['status'] = 'Unknown'
            else:
                conditions['Ready']['status'] = 'True'
                conditions['GuacReady']['message'] = f'There are {len(guacamole_sessions)} / {max_guacamole_sessions} guacamole sessions'
                if len(guacamole_sessions) < max_guacamole_sessions:
                    conditions['GuacReady']['status'] = 'True'
                else:
                    conditions['GuacReady']['reason'] = 'Limit'
                if any(any(condition.type == 'Active' and condition.status == 'True' for condition in guacamole_session.conditions) for guacamole_session in guacamole_sessions):
                    conditions['GuacActive']['status'] = 'True'

        assert all(condition['status'] in ('True', 'False', 'Unknown') for condition in conditions.values()), conditions

        req.context.result = {
            'conditions': conditions.values(),
        }
