"""
This module provides REST handlers for Device Name
"""
import falcon.status_codes as http_status
from central_ag import devices_names
from central_ag.api import (
    schema,
    authz,
)
from netops import automation_gateway as ag
from typing import Optional


class DevicesNames:
    """
    Handlers for updating and reading devices' customized names.
    """
    schema = schema.DeviceNameList()

    def __init__(
            self,
            *,
            ag_devices: Optional[ag.DevicesStub] = None,
    ):
        self._ag_devices = ag_devices

    def on_get(self, request, response):
        """
        Devices' names GET endpoint.
        ---
        description: Retrieve all of the devices' names, and return them as a list.
        responses:
            200:
                description: Successfully retrieved
                content:
                    application/json:
                        schema: DeviceNameList
        """
        if not authz.can(request.context.get("token"), "/nom/ag/devices_names", "get"):
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
            return

        request.context.result = devices_names.load()

    def on_put(self, request, response):
        """
        Devices' names PUT endpoint.
        ---
        description: Save or update the names of devices.
        requestBody:
            content:
                application/json:
                    schema: DeviceNameList
        responses:
            204:
                description: Successfully updated
            400:
                description: Invalid request
            403:
                description: Permission denied
        """
        # Validate the request
        manual_mac_name = {}
        token = request.context.get("token")
        for device in request.context.json["devices_names"]:
            mac = device.get("mac", None)
            name = device.get("name", None)
            if mac is None or name is None:
                response.status = http_status.HTTP_400
                response.media = {"error": "Invalid request - Should contain both Device Mac and Name in request"}
                return

            if mac in manual_mac_name:
                # Check if setting different names for the same device
                response.status = http_status.HTTP_400
                response.media = {"error": f"Invalid request - Duplicated names defined for device: '{mac}'"}
                return
            # Add to the groups list to check if the group has been defined before
            manual_mac_name[mac] = name
        if authz.can(token, "/nom/ag/devices_names", "put"):
            devices_names.store(request.context.json)
            self._ag_devices.Publish(ag.DevicesPublish.Request(manual_mac_name=ag.Manual.MacName(data=manual_mac_name)))
            response.status = http_status.HTTP_204
        else:
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
            return
