"""
the ztp module is the interface with the functionality for generating ZTP
services configuration, and writing those out to appropriate directories for
deployment via ansible. The ZTP templates are based on the vendor type of the
resource bundles.
"""
import os
import errno
import hashlib
import codecs
import jinja2 as jin
from jinja2 import Environment, FileSystemLoader
from central_dop.config import strip_extension


class Error(Exception):
    """Base class for exceptions in the ZTP module."""

    def __init__(self, msg):  # pylint: disable=super-init-not-called
        self._message = msg

    @property
    def message(self):
        """ return the exception's message """
        return self._message


class GenerateError(Error):
    """
    GenerateError is raised when the handler functions fail to create the
    resource files.
    """


class ResourceFileError(Error):
    """
    ResourceFileError is raised when the resource files have not been uploaded.
    """


class ScriptFileError(Error):
    """
    ScriptFileError is raised when the script files have not been uploaded.
    """


class DeviceUnsupportedError(Error):
    """
    DeviceUnsupported is raised when an invalid device type is provided in
    config.
    """


class BaseHandler:  # pylint: disable=too-many-instance-attributes,too-many-public-methods
    """
    The base class for all of the vendor-specific handlers. Most of the child
    classes will be similar but there are some specific overrides for the
    vendor weirdness.
    """

    def __init__(self, template_dir="/templates"):
        self._config_file = None
        self._image_file = None
        self._script_file = None
        self._post_provision_script = None
        self._bundle_name = None
        self._vendor = None
        self._mac_addresses_allowed = []
        self._mac_addresses_disallowed = []
        self._serial_numbers = []
        self._template_dir = template_dir
        self._base_dir = None
        self._env = Environment(
            loader=FileSystemLoader(template_dir),
            trim_blocks=True,
            lstrip_blocks=True
        )
        self._env.globals["encode_cisco_ios_image_file"] = encode_cisco_ios_image_file
        self._other_devices = []

    @property
    def config_file(self):
        """ the config file specific to this config """
        return self._config_file

    @config_file.setter
    def config_file(self, value):
        """ set the config file specific to this config """
        self._config_file = value

    @property
    def image_file(self):
        """ the image file specific to this config """
        return self._image_file

    @image_file.setter
    def image_file(self, value):
        """ set the image file specific to this config """
        self._image_file = value

    @property
    def script_file(self):
        """ the script file specific to this config """
        return self._script_file

    @script_file.setter
    def script_file(self, value):
        """ set the script file specific to this config """
        self._script_file = value

    @property
    def post_provision_script(self):
        """ the script to run after the devices are provisioned """
        return self._post_provision_script

    @post_provision_script.setter
    def post_provision_script(self, value):
        """ set the script that will run after the devices are provisioned """
        self._post_provision_script = value

    @property
    def bundle_name(self):
        """ the name of the resource bundle """
        return self._bundle_name

    @bundle_name.setter
    def bundle_name(self, value):
        """ set the name of the resource bundle """
        self._bundle_name = value

    @property
    def mac_addresses_allowed(self):
        """
        A list of mac addresses that are whitelisted for provisioning of
        the configuration.
        """
        return self._mac_addresses_allowed

    @mac_addresses_allowed.setter
    def mac_addresses_allowed(self, values):
        """
        Replace the current list of whitelisted mac addresses with the given
        list.
        """
        self._mac_addresses_allowed = values

    @property
    def mac_addresses_disallowed(self):
        """
        A list of mac addresses that are blacklisted for provisioning of
        the configuration.
        """
        return self._mac_addresses_disallowed

    @mac_addresses_disallowed.setter
    def mac_addresses_disallowed(self, values):
        """
        Replace the current list of blacklisted mac addresses with the given
        list.
        """
        self._mac_addresses_disallowed = values

    @property
    def serial_numbers(self):
        """
        A list of whitelisted serial numbers for provisioning of the configuration.
        """
        return self._serial_numbers

    @serial_numbers.setter
    def serial_numbers(self, values):
        """
        Replace the current list of serial numbers with a new list.
        """
        self._serial_numbers = values

    @property
    def template_dir(self):
        """ the directory name which contains the templates for the vendor """
        return self._template_dir

    @template_dir.setter
    def template_dir(self, value):
        """ set the directory which contains the templates for the vendor """
        self._template_dir = os.path.abspath(value)

    @property
    def base_dir(self):
        """ the directory which contains all of the base templates """
        return self._base_dir

    @base_dir.setter
    def base_dir(self, value):
        """ set the directory which contains all of the base templates """
        self._base_dir = value

    @property
    def vendor(self):
        """ the name of the vendor for the configs """
        return self._vendor

    @vendor.setter
    def vendor(self, value):
        """ set the name of the vendor for the configs """
        self._vendor = value

    @property
    def env(self):
        """ get the current environment """
        return self._env

    @property
    def other_devices(self):
        """
        Other devices in during the deployment is specific to Cloud Provisioned Device type
        where we support model-specific configurations.
        """
        return self._other_devices

    @other_devices.setter
    def other_devices(self, values):
        """ sets the _other_devices for the Opengear configs """
        self._other_devices = values

    def process_device_resources(self, device_resources, name):
        """
        this sets up the handler with the provided device resources and name
        @device_resources - a dict containing the configuration supported in the
            resource bundles
        @name - the name of the resource bundle
        """
        self.bundle_name = name
        self.config_file = device_resources.get("config_file")
        self.image_file = device_resources.get("image_file")
        self.script_file = device_resources.get("script_file")
        if "mac_address" in device_resources:
            self.process_mac_addresses(device_resources["mac_address"])
        if "serial_number" in device_resources:
            self.serial_numbers = device_resources["serial_number"]
        if "post_provision_script" in device_resources:
            self.post_provision_script = device_resources.get("post_provision_script")

    def process_other_device_resources(self, other_resource_bundles):
        """
        this sets up the handler with the provided device resources and name
        @device_resources - a dict containing the configuration supported in the
            resource bundles
        """
        bundle_list = []
        if other_resource_bundles:
            for bundle_key, bundle_item in other_resource_bundles.items():
                handler = get_handler_for_device(
                    bundle_item.get("device_type"))
                handler.process_device_resources(bundle_item, bundle_key)
                # CiscoXR has extra Parameter for get_data().
                if bundle_item.get("device_type") == "cisco_xr":
                    mode = None
                    if bundle_item.get("image_file"):
                        mode = "ipxe"
                    if bundle_item.get("config_file"):
                        # Cisco XR supports a config file OR script file with the same DHCP option
                        mode = "cfg"
                    # pylint: disable=too-many-function-args
                    bundle_list.append(handler.get_data(mode))
                else:
                    bundle_list.append(handler.get_data())
            if bundle_list:
                self.other_devices = bundle_list

    def copy_scripts(self, root_dir, deploy_dir):
        """
        Copy any resource bundle scripts(currently only post-provisioning script)
        into the deployment directory.
        """
        files = [self.post_provision_script]

        for f in files:
            if f:
                try:
                    # Look in scripts directory
                    file_from = "{}/scripts/{}".format(root_dir, f)
                    file_to = "{}/scripts/{}".format(deploy_dir, f)
                    if os.path.isfile(file_from):
                        os.symlink(file_from, file_to)
                    else:
                        raise ScriptFileError("resource file '{}' not found".format(f))
                except OSError as err:
                    # catch file exists error, which can happen if the user
                    # has specified the same file be used for 2 different
                    # resource groups.
                    if err.errno != errno.EEXIST:
                        raise err

    def copy_resources(self, root_dir, deploy_dir): # pylint: disable=unused-argument
        """
        Placeholder function
        """
        return

    def render_template(self, template, data):
        """
        renders the jinja2 template and returns it as a string
        @template - the name of the template(in the self.template_dir)
        @data - the required data for the template
        """
        return self.env.get_template(template).render(data=data)

    def write_vendor_class_template(self, filename, data):
        """
        renders the vendor class template and writes it to the given file.
        @filename - the name of the output file
        @data - the required data for the template
        """
        # all vendors support mac address matching for options, setup the data
        # input to the template.
        self.set_mac_address(data)
        with open(filename, "w+") as file:
            file.write(self.render_template("vendor-class-template.j2", data))

    def process_mac_addresses(self, mac_addresses):
        """
        this function handles converting the mac addresses provided by the user
        into the dhdpd expected format. After converting the addresses, the
        allowed & disallowed lists are filled with the appropriate macs.
        @mac_addresses - a list of mac addresses as specified in a device resource
            bundle
        """
        # Mac addresses are validated prior to getting to this step. dhcpd is
        # very particular about substring matching. See dhcp-eval(5) about the
        # below transformations.
        # In short:
        # - strip *(its metadata for our parsing), remove leading zeroes,
        # and add a leading 1: to indicate it's an ethernet mac address.
        for mac in mac_addresses:
            mac = mac.replace("*", "").lower()
            tmp = ''
            for m in mac.split(":"):
                if len(m) == 2 and m[0] == '0':
                    tmp = tmp + m[1] + ":"
                elif len(m) == 3 and m[1] == '0':
                    tmp = tmp + m[0] + m[2] + ":"
                elif len(m) != 0:  # pylint: disable=len-as-condition
                    tmp = tmp + m + ":"
            mac = tmp[:len(tmp) - 1]
            mac = "1:" + mac

            if mac.find("!") != -1:
                mac = mac.replace("!", "")
                self.mac_addresses_disallowed.append({
                    'mac': mac,
                    'length': len(mac.split(":")),
                })
            else:
                self.mac_addresses_allowed.append({
                    'mac': mac,
                    'length': len(mac.split(":")),
                })

    def set_mac_address(self, data):
        """
        this is used to setup the data dict provided with the mac addresses for
        the handler
        @data - the dict passed into the templates
        """
        if self.mac_addresses_allowed:
            data["mac_addresses_allowed"] = self.mac_addresses_allowed
        if self.mac_addresses_disallowed:
            data["mac_addresses_disallowed"] = self.mac_addresses_disallowed

    def get_data(self):
        """
        returns a dict as required by the dhcp templates. This can be overridden
        by subclasses that require different field names.
        Any templated filenames will be stripped of the .j2 extension here, so that the DHCP
        configs will serve the actual filenames to downstream devices. The remote HTTP and TFTP
        file servers will handle these requests and serve the appropriate files.
        """
        data = {}
        data['config_file'] = strip_extension(self.config_file)
        data['image_file'] = strip_extension(self.image_file)
        data['name'] = self.bundle_name
        data['vendor'] = self.vendor
        data['script_file'] = strip_extension(self.script_file)
        data['mac_addresses_allowed'] = self.mac_addresses_allowed
        data['mac_addresses_disallowed'] = self.mac_addresses_disallowed
        data['post_provision_script'] = self.post_provision_script
        data['other_devices'] = self.other_devices
        return data


class AristaHandler(BaseHandler):
    """
    Arista specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "arista"

    def get_data(self):
        """
        override to include the boot_file key
        """
        data = super().get_data()
        data['boot_file'] = "{}_boot.sh".format(self.bundle_name)
        return data

    def generate_templates(self, base_dir):
        """
        Arista requires a boot script, which is responsible for loading
        config/image files. Create this first, making sure to mark it as template.
        """
        data = self.get_data()
        with open("{}/downloads/{}.j2".format(base_dir, data['boot_file']), 'w+') as file:
            file.write(self.render_template("arista-bootscript.j2", data))
        self.write_vendor_class_template(
            "{}/dhcpd/{}".format(base_dir, self.bundle_name), data)


class ArubaHandler(BaseHandler):
    """
    Aruba specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "aruba"

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        self.write_vendor_class_template(
            "{}/dhcpd/{}".format(base_dir, self.bundle_name), self.get_data())


class CiscoHandler(BaseHandler):
    """
    Cisco IOS specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "cisco"

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        self.write_vendor_class_template(
            "{}/dhcpd/{}".format(base_dir, self.bundle_name), self.get_data())


class CiscoNXHandler(BaseHandler):
    """
    Cisco NX-OS specific overrides for the vendor differences.
    """
    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "cisco_nx"

    def get_name(self, data, key):
        """
        Function to handle string value retrieving from a disctionary and replacing
        None entries with an empty string for later concatenation.
        """
        if key in data.keys():
            file = data[key]
            if file is not None:
                return file
        return ''

    def default_boot_script(self, data):
        """
        Returns the first 16 characters of a sha256 for the combination of resource
        file names.
        """
        if 'boot_file' in data.keys():
            boot_file = data['boot_file']
            if boot_file is not None:
                return boot_file
        config_file = self.get_name(data, 'config_file')
        image_file = self.get_name(data, 'image_file')
        return hashlib.sha256((config_file + ':' + image_file).encode('utf-8')).hexdigest()[:16]+".py"

    def get_data(self):
        """
        override to include the boot_file key
        """
        data = super().get_data()
        data['boot_file'] = self.default_boot_script(data)
        return data

    def generate_templates(self, base_dir):
        """
        Cisco  NX-OS POAP requires a boot script, which is responsible for loading
        config/image files. Create this first, making sure to mark it as template.
        """
        data = self.get_data()
        with open("{}/downloads/{}.j2".format(base_dir, data['boot_file']), 'w+') as file:
            file.write(self.render_template("nxos-bootscript.j2", data))
        self.write_vendor_class_template(
            "{}/dhcpd/{}".format(base_dir, self.bundle_name), data)


class CiscoXEHandler(BaseHandler):
    """
    Cisco IOS XE specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "cisco_xe"

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        self.write_vendor_class_template(
            "{}/dhcpd/{}".format(base_dir, self.bundle_name), self.get_data())


class CiscoXRHandler(BaseHandler):
    """
    Cisco IOS XR specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "cisco_xr"

    def get_data(self, mode):  # pylint: disable=arguments-differ
        """
        override for the cisco_xr specific options
        @mode - should be either ipxe or cfg to work with devices
        """
        data = super().get_data()
        data['name'] = "{}_{}".format(self.bundle_name, mode)
        data['vendor'] = "{}_{}".format(self.vendor, mode)

        if mode == "ipxe" and self.serial_numbers:
            # The image_file for iPXE supports url string substitution.
            data['image_file'] = "${{serial:uristring}}-{}".format(
                self.image_file)

        return data

    def generate_templates(self, base_dir):
        """
        XR is handled a bit differently as it supports image via iPXE and
        config/script via ZTP so we handle it as 2 separate DHCP classes.
        """
        if self.image_file:
            self.write_vendor_class_template(
                "{}/dhcpd/{}_ipxe".format(base_dir, self.bundle_name), self.get_data("ipxe"))

        if self.config_file:
            # Cisco XR supports a config file OR script file with the same DHCP option
            self.write_vendor_class_template(
                "{}/dhcpd/{}_cfg".format(base_dir, self.bundle_name), self.get_data("cfg"))

    def copy_resources(self, root_dir, deploy_dir):
        """
        copy_resources sets up the symlinks for a cisco_xr device type when serial
        numbers have been specified
        """
        files = [file for file in
                 [self.config_file, self.image_file, self.script_file] if file]

        for f in files:
            try:
                # Create a symlink of the file for each of the serial numbers,
                # as the device will substitute their serial number for the
                # request for the image file. The serial number will prepend
                # the original image file name.
                if self.serial_numbers and (f == self.image_file):
                    for serial in self.serial_numbers:
                        file_from = "{}/downloads/{}".format(deploy_dir, f)
                        file_to = "{}/downloads/{}-{}".format(
                            deploy_dir, serial, f)
                        if os.path.isfile(file_from):
                            os.symlink(file_from, file_to)
                        else:
                            raise ResourceFileError(
                                "resource file '{}' not found".format(f))
            except OSError as err:
                # catch file exists error, which can happen if the user
                # has specified the same file be used for 2 different
                # resource groups.
                if err.errno != errno.EEXIST:
                    raise err


class CloudProvisionedHandler(BaseHandler):
    """
    Cloud Provisioned Device specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "cloud_provisioned"

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        # preformat other_resources
        self.write_vendor_class_template(
            "{}/dhcpd/{}".format(base_dir, self.bundle_name),
            self.get_data())


class CumulusHandler(BaseHandler):
    """
    Cumulus specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "cumulus"

    def get_data(self):
        """
        override to include the boot_file key
        """
        data = super().get_data()
        data['boot_file'] = "{}_boot.cumulus.sh".format(self.bundle_name)
        return data

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        data = self.get_data()
        # Cumulus requires a bootscript for ordered provisioning
        with open("{}/downloads/{}.j2".format(base_dir, data['boot_file']), 'w+') as file:
            file.write(self.render_template("cumulus-bootscript.j2", data))
        self.write_vendor_class_template(
            "{}/dhcpd/{}".format(base_dir, self.bundle_name), data)


class HuaweiHandler(BaseHandler):
    """
    Huawei specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "huawei"

    def get_data(self):
        """
        overrides the data to include a boot_file option
        """
        data = super().get_data()
        data['boot_file'] = "{}.ini".format(self.bundle_name)
        return data

    def generate_templates(self, base_dir):
        """
        Huawei requires a boot .ini file, which is responsible for specifying
        config/image files. Create this first.
        """
        data = self.get_data()
        with open("{}/downloads/{}".format(base_dir, data['boot_file']), "w+") as file:
            file.write(self.render_template("huawei-bootinit.j2", data))
        # Now create the DHCP template.
        self.write_vendor_class_template("{}/dhcpd/{}".format(base_dir, self.bundle_name), data)


class JuniperHandler(BaseHandler):
    """
    Juniper specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "juniper"

    def get_data(self):
        """
        overrides the data to include transfer_mode and image_filetype if an
        image file was included
        """
        data = super().get_data()

        if self.image_file:
            data['image_filetype'] = 'file'
        return data

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        self.write_vendor_class_template("{}/dhcpd/{}".format(base_dir, self.bundle_name), self.get_data())


class OpengearHandler(BaseHandler):
    """
    Opengear specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "opengear"
        self._model = None
        self._enrollment = {
            "url": None,
            "bundle": None,
            "token": None,
            "port": None,
        }

    @property
    def model(self):
        """
        model is specific to Opengear where we support model-specific
        configurations.
        """
        return self._model

    @model.setter
    def model(self, value):
        """ sets the model for the Opengear configs """
        self._model = value

    @property
    def enrollment(self):
        """ enrollment is data for the downstream device to enrol into Lighthouse """
        return self._enrollment

    @enrollment.setter
    def enrollment(self, value):
        """ set the enrollment data for the opengear configs """
        self._enrollment = value

    def process_device_resources(self, device_resources, name):
        """
        override the base function to add support for defining the model and
        lighthouse enrollment configuration specific to Lighthouse compatible
        devices(i.e. Opengear console servers).
        @device_resources - a dict containing the configuration supported in the
            resource bundles
        @name - the name of the resource bundle
        """
        super().process_device_resources(device_resources, name)
        self.model = device_resources.get("model")
        if device_resources.get("enrollment"):
            self.enrollment = device_resources.get("enrollment")

    def get_data(self):
        """
        overrides the data to include model, which only Opengear devices support
        so far
        """
        data = super().get_data()
        data['model'] = self.model
        data['enrollment'] = self.enrollment
        return data

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        self.write_vendor_class_template("{}/dhcpd/{}".format(base_dir, self.bundle_name), self.get_data())


class PicaHandler(BaseHandler):
    """
    Pica specific overrides for the vendor differences.
    """

    def __init__(self, template_dir="/templates"):
        BaseHandler.__init__(self, template_dir)
        self.vendor = "pica"

    def generate_templates(self, base_dir):
        """ vendor specific for template generation """
        self.write_vendor_class_template("{}/dhcpd/{}".format(base_dir, self.bundle_name), self.get_data())


def get_handler_for_device(device_type, template_dir="/templates"):
    """
    returns the handler for the given vendor
    @device_type - the vendor for the device
    @template_dir - an optional argument for the handler constructor
    """
    handler = None
    if device_type == "aruba":
        handler = ArubaHandler(template_dir)
    elif device_type == "arista":
        handler = AristaHandler(template_dir)
    elif device_type == "cisco":
        handler = CiscoHandler(template_dir)
    elif device_type == "cisco_nx":
        handler = CiscoNXHandler(template_dir)
    elif device_type == "cisco_xe":
        handler = CiscoXEHandler(template_dir)
    elif device_type == "cisco_xr":
        handler = CiscoXRHandler(template_dir)
    elif device_type == "cloud_provisioned":
        handler = CloudProvisionedHandler(template_dir)
    elif device_type == "cumulus":
        handler = CumulusHandler(template_dir)
    elif device_type == "huawei":
        handler = HuaweiHandler(template_dir)
    elif device_type == "juniper":
        handler = JuniperHandler(template_dir)
    elif device_type == "opengear":
        handler = OpengearHandler(template_dir)
    elif device_type == "pica":
        handler = PicaHandler(template_dir)
    else:
        raise DeviceUnsupportedError(
            "device type ({}) is not supported".format(device_type))

    return handler


def handle_deployment(root_dir, inventory, deployment_list, device_resources, template_dir="/templates"):
    """
    this is the function that glues all of the other functions together. basically sets
    up the handler, generates resources, and copies the expected files for the
    given resource bundle.
    """
    base_dir = "{}/{}".format(root_dir, inventory)
    dirs = ["/dhcpd", "/downloads", "/scripts"]
    for d in dirs:
        if not os.path.isdir(base_dir + d):
            os.makedirs(base_dir + d)

    for resource_bundle in deployment_list.get(inventory):
        resource = device_resources.get(resource_bundle)
        handler = get_handler_for_device(
            resource.get("device_type"), template_dir)
        handler.process_device_resources(resource, resource_bundle)

        other_resources = {}
        for bundle_key, bundle_item in device_resources.items():
            if bundle_key in deployment_list.get(inventory) and bundle_key != resource_bundle:
                other_resources[bundle_key] = bundle_item
        handler.process_other_device_resources(other_resources)

        try:
            handler.generate_templates(base_dir)

        except jin.exceptions.TemplateNotFound:
            raise GenerateError("failed to generate configuration templates") from None
        handler.copy_scripts(root_dir, base_dir)
        handler.copy_resources(root_dir, base_dir)


def generate_ztp_resources(deploy_dir, config, template_dir="/templates"):
    """the main entry function for the ztp module. All the caller needs to
    provide is a config and directory to create the deployment."""
    for group_name in list(config.get('deployment').keys()):
        handle_deployment(
            deploy_dir,
            group_name,
            config.get('deployment'),
            config.get('device_resources'),
            template_dir,
        )


def encode_cisco_ios_image_file(filename):
    """
    generates a hex string for use in dhcpd.conf option 125 for cisco's
    "autoinstall", after first converting it to the assumed name of a
    text file that will be created by the hostgroup tasks (ansible, after
    copying the resources to the remote, at least currently)

    :param filename: the image resource file name (.bin or .tar)
    :return: encoded string e.g.
             00:00:00:09:0A:05:13:66:72:65:65:7A:74:70:5F:69:6F:73:5F:75:70:67:72:61:64:65
    """
    # convert the filename into the assumed filename of a text file containing
    # the original value of filename that is also served via tftp...
    filename = hashlib.md5(filename.encode('utf-8')).hexdigest() + ".filename.txt"
    # again encode the filename (twice more actually), first into the strange
    # format expected by the cisco device, then into the hex string format
    # supported by dhcpd.conf
    filename = codecs.encode(filename.encode("ascii"), encoding="hex").decode("ascii")
    length = hex(int(len(filename) / 2))[2:]
    while len(length) < 2:
        length = "0" + length
    return ":".join(map("".join, zip(*[iter("000000090a05" + length + filename)] * 2))).upper()
