"""
nodes is a module providing some generally useful functions for communicating
with a lighthouse.
"""
import os
import subprocess
import requests

requests.urllib3.disable_warnings()


class Error(Exception):
    """
    base class for exceptions
    """

    def __init__(self, msg):  # pylint: disable=super-init-not-called
        self._message = msg

    @property
    def message(self):
        """ returns the exception's error message. """
        return self._message


class Unauthorized(Error):
    """
    raised when the certificate is not authorized to make any request
    """


class Unknown(Error):
    """
    raised when the request failed for an unknown reason, e.g. InternalServerError
    """


class Forbidden(Error):
    """
    raised when the request is not allowed to access the resource
    """


class NotFound(Error):
    """
    raised when the resource was not found on the server
    """


class Invalid(Error):
    """
    raised when the request contained invalid data
    """


def get_host_address():
    """
    returns the address of the host based on the default gateway of the
    underlying docker network (which is currently be the host address)
    """
    route_info = subprocess.check_output(["/sbin/ip", "route", "show", "default"])
    addr = route_info.split()[2].decode("utf-8")
    return addr


# Cache value
apiVersion = None


def get_api_version():
    """
    returns the current version of the LH api
    """

    global apiVersion

    if apiVersion is None:
        apiVersion = os.environ.get("LH_API_VERSION")

    # Fallback option is v3.2
    if apiVersion is None:
        apiVersion = "v3.2"

    return apiVersion


def get_certs():
    """
    get_certs returns the ssl public & private keypair for the requests lib
    to use.
    """
    return "/root/.netops/lhvpn/lhvpn_server.crt", "/root/.netops/lhvpn/lhvpn_server.key"


def make_request(url, token=None):
    if not token:
        response = requests.get(url, verify=False, cert=get_certs())
    else:
        response = requests.get(url, verify=False, headers={"Authorization": f"Token {token}"})
    status_code = response.status_code

    if status_code < 300:
        return response.json()

    if status_code == 400:
        raise Invalid("invalid request body: {}".format(url))
    if status_code == 401:
        raise Unauthorized("request was unable to be authenticated: {}".format(url))
    if status_code == 404:
        raise NotFound("requested resource was not found: {}".format(url))
    if status_code == 403:
        raise Forbidden("access to requested resource denied: {}".format(url))

    raise Unknown("unknown error making request: {}".format(url))


def resolve_smartgroup_to_ids(smartgroup):
    """
    resolve_smartgroup_to_ids returns the list of node ids for a smartgroup.
    Currently the response from _resolve_smartgroup is returned directly, however
    we may want to add data to the response in the future so we'll keep this as
    a separate function for now.
    """
    node_ids = []
    q_smartgroup = requests.utils.quote(smartgroup)
    url = "https://{}/api/{}/nodes/smartgroups/{}/nodes?by=name".format(
        get_host_address(), get_api_version(), q_smartgroup)
    response = make_request(url)
    if 'nodes' in response:
        node_ids = response['nodes']
    return node_ids
