import logging
from . import device

# Device states
UNKNOWN_STATE = 'unknown'
SETUP_STATE = 'setup'
PREAUTH_STATE = 'preauth'
PROMPT_STATE = 'prompt'
ENABLED_STATE = 'enabled'


class StateMachine():
    def __init__(self, results, expect):
        self.results = results
        self.expect = expect
        self.logger = logging.getLogger()
        self.current_state = UnknownState(self.results, self.expect)

    def next_state(self):
        if self.current_state is None:
            return None
        self.logger.debug("Moving from {} to next state".format(self.current_state))
        if str(self.current_state) != UNKNOWN_STATE:
            # Can only move to next state if we know what state we're currently in
            self.current_state.next()
        for state in self.current_state.possible():
            s = state(self.results, self.expect)
            if s.test():
                self.logger.debug("Device in state: {}".format(s))
                self.current_state = s
                break
            self.logger.debug("Test for state {} failed".format(s))
        # TODO handle case where failed to change state?
        return self.current_state


class State():
    """
    The State class models device states and how to transition between them.

    A State implementation knows how to test whether the device is in this state, and it
    knows how to transition OUT of it's current state but NOT necessarily which state will result.
    Thus, the state workflow should be:

    1. Test possible states to see which state device is currently in
    2. Move out of current state
    3. Get possible resulting states
    4. Repeat to get to target state

    """
    def __init__(self, results, expect):
        self.results = results
        self.expect = expect

    def test(self):
        """
        This function should return True if the conditions for this state are met
        """
        assert 0, 'not implemented'

    def next(self):
        """
        This function should run whatever action(s) required to move out of this state
        """
        assert 0, 'not implemented'

    def possible(self):
        """
        This function should return (in order of checking priority) the possible states that the port/device
        can be in after moving out of this state.
        """
        assert 0, 'not implemented'


class UnknownState(State):
    """
    This is the initial state where the actual device state is not known
    """
    def possible(self):
        return [EnabledState, PromptState, PreauthState, SetupState]

    def __str__(self):
        return UNKNOWN_STATE


class SetupState(State):
    """
    The 'Setup' state represents a device that is blocked on an initial setup dialog, e.g.
    to configure a root user.
    """
    def test(self):
        return device.check_for_setup_dialog(self.expect)

    def next(self):
        device.attempt_exit_setup(self.results, self.expect)

    def possible(self):
        return [EnabledState, PromptState, PreauthState, SetupState]

    def __str__(self):
        return SETUP_STATE


class PreauthState(State):
    """
    The 'Preauth' state represents a device that is on a login prompt waiting for credentials
    e.g. "Username:".
    """
    def test(self):
        return device.check_for_preauth(self.expect)

    def next(self):
        device.attempt_default_logins(self.results, self.expect)

    def possible(self):
        return [EnabledState, PromptState, PreauthState]

    def __str__(self):
        return PREAUTH_STATE


class PromptState(State):
    """
    The 'Prompt' state represents a device that has a usable prompt.
    """
    def test(self):
        return device.check_for_prompt(self.expect)

    def next(self):
        device.attempt_enable(self.expect)

    def possible(self):
        return [EnabledState, PromptState]

    def __str__(self):
        return PROMPT_STATE


class EnabledState(State):
    """
    The 'Enabled' state represents a device that is 'enabled', i.e can run advanced commands.
    This is the desired state for most of our discovery.
    """
    def test(self):
        return device.check_for_enabled(self.expect)

    def next(self):
        return

    def possible(self):
        return [EnabledState]

    def __str__(self):
        return ENABLED_STATE
