import time
import subprocess
import select
import pexpect
import os
import secrets
import string

# Device types
UNKNOWN = "unknown"
ARISTA = "arista"
CISCO_IOS = "cisco_ios"
CISCO_XE = "cisco_xe"
CISCO_XR = "cisco_xr"
OPENGEAR_OG = "opengear_console_server"
OPENGEAR_NG = "opengear_operations_manager"

# Default bauds to discover on, in order of how prevalent they are in 3rd party devices
DEFAULT_BAUDS = ['9600', '57600', '115200']

# Default pinouts to discover on
DEFAULT_PINOUTS = ['X2', 'X1']

def ready(port, timeout=120):
    """
    Return True when device hasn't sent any data for a 'reasonable' amount of time
    Return False if timeout is reached
    """
    # How many seconds we wait before deeming the port is 'ready'
    secs_til_ready = 15

    timeout_time = time.time() + timeout
    proc = subprocess.Popen('/bin/pmshell -l port{}'.format(port).split(), stdin=subprocess.PIPE, stdout=subprocess.PIPE)  # pylint: disable=consider-using-with
    secs_counter = 0

    port_ready = False
    while time.time() < timeout_time:
        r, _, _ = select.select([proc.stdout], [], [], 0)

        if proc.stdout in r:
            secs_counter = 0
            os.read(proc.stdout.fileno(), 1024)
        else:
            secs_counter += 1
            if secs_counter >= secs_til_ready:
                port_ready = True
                break
            time.sleep(1)
    proc.kill()
    return port_ready

def prepare_discovery(expect, device_type):
    """
    Perform any setup tasks required before full discovery.

    Must be in the target state i.e. Enabled
    """
    if device_type in (CISCO_IOS, CISCO_XE):
        expect.send("terminal length 0\r\n")
    if device_type == OPENGEAR_OG:
        # TODO line feeds not working on OGCS, work around this by sending a reset for now
        expect.send("reset\r\n")

def check_for_setup_dialog(expect):
    """
    Return True if device is in an initial setup dialog
    """
    setup_patterns = [
        # TODO Cisco XR specific, make this more generic
        'Enter root-system username:',
        # Cisco XE
        'Would you like to enter the initial configuration dialog',
    ]
    try:
        expect.send('\r\n')
        matched = expect.expect(setup_patterns)
        if 0 <= matched <= len(setup_patterns):
            return True
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        return False
    return False

def check_for_preauth(expect):
    """
    Return True if device requires an auth login, e.g. 'localhost login:'
    """
    patterns = [
        r'\S+ login:',
        'Username:'
    ]
    try:
        expect.send('\r\n')
        matched = expect.expect(patterns)
        if 0 <= matched <= len(patterns):
            return True
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        return False
    return False

def attempt_exit_setup(results, expect):
    """
    Attempt to exit a device's initial setup prompt and get to the pre-auth
    or prompt state.
    TODO maybe it's better to pass the matched prompt in to this function
    instead of finding it again
    """
    setup_patterns = [
        'Enter root-system username:',
        'Would you like to enter the initial configuration dialog',
    ]

    try:
        expect.send('\r\n')
        matched = expect.expect(setup_patterns)
        if matched == 0:
            # Cisco XR
            # The only way to get out of this prompt is by creating a user.
            # Generate a username and password, and store them for later use.
            # TODO do we need to do more here?
            password = ''.join(secrets.choice(string.ascii_letters + string.digits) for i in range(12))
            expect.send(f'netops\n{password}\n{password}\n\r')

            # Set credentials so they can be stored later
            results['discovery_username'] = "netops"
            results['discovery_password'] = password
        # Cisco XE
        elif matched == 1:
            expect.send('no\n\r')
            expect.expect('Would you like to terminate autoinstall:')
            expect.send('yes\n\r')
        else:
            return
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        return
    return

def check_for_prompt(expect):
    """
    Return True if device has a prompt, e.g. 'Switch>'
    """
    prompts = [
        r'>\s*$'
    ]

    for _ in range(2):
        try:
            expect.send('\r\n')
            matched = expect.expect(prompts)
            if matched >= 0 or matched <= len(prompts):
                return True
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            continue
    return False

def check_for_enabled(expect):
    """
    Return True if device has an 'enabled' prompt, e.g. 'Switch#'
    """
    prompts = [
        r'#\s*$'
    ]

    for _ in range(2):
        try:
            expect.send('\r\n')
            matched = expect.expect(prompts)
            if matched >= 0 or matched <= len(prompts):
                return True
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            continue
    return False

def attempt_enable(expect):
    """
    Attempt to enable a device.
    """
    try:
        expect.send('enable\r\n')
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        return None
    return None

def attempt_default_logins(results, expect):
    """
    This function tries to login with default credentials,
    and returns True if it succeeds to get a prompt.
    """
    default_credentials = [
        {
            "username": "root",
            "password": "default"
        },
        # Leave this one at the end, as the empty password leaves a bad state
        # e.g. "Password:" prompt on XE
        {
            "username": "admin",
            "password": ""
        },
    ]
    for creds in default_credentials:
        expect.send('\r\n')
        attempt_login(expect, creds)
        if check_for_prompt(expect) or check_for_enabled(expect):
            return True
        # Delay in here as it may take some time to attempt to check the password

    # If any credentials have been set as part of discovery, attempt them
    if results.get('discovery_username', None):
        creds = {
            'username': results.get('discovery_username', ''),
            'password': results.get('discovery_password', '')
        }
        attempt_login(expect, creds)
        if check_for_prompt(expect):
            return True

    return False

def attempt_login(expect, credentials):
    """
    Attempt to login by providing credentials.
    This is currently specifically for Arista which requires
    a default 'admin' login.
    This function doesn't check the response, it is up to the caller
    to further check for login success.

    TODO handle user-provided credentials
    """
    expect.send('\r\n')
    time.sleep(5)
    expect.send("{}\n".format(credentials['username']))
    time.sleep(2)
    # TODO this might need to be more robust, e.g. expecting a password prompt
    expect.send("{}\n".format(credentials['password']))

def attempt_exit(expect):
    """
    Attempt to exit the current device session
    """
    # Send 'exit' to leave any active session
    expect.send('\r\nexit\r\nexit\r\n\r\n\r\n')
    time.sleep(2)

    # Send Ctrl-C code to try and exit any running processes
    expect.send('\003\003\003\003')

    try:
        expect.expect(r'\S+')
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        return

def get_device_type(expect):
    """
    Determine device type by running a CLI command and matching output.

    This function assumes the device is in the correct state to be able to run commands.
    """
    device_type = UNKNOWN

    # Arista/Cisco device type detection
    device_type_matches = ["Arista", "Cisco IOS Software", "Cisco IOS XE Software", "Cisco IOS XR Software"]
    try:
        expect.send("show version\r\n")
        # TODO make this a bit nicer
        matched = expect.expect(device_type_matches)
        if matched == 0:
            device_type = ARISTA
        if matched == 1:
            device_type = CISCO_IOS
        if matched == 2:
            device_type = CISCO_XE
        if matched == 3:
            device_type = CISCO_XR
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        device_type = UNKNOWN

    # Opengear device type detection
    if device_type == UNKNOWN:
        try:
            expect.send("cat /etc/version\r\n")
            device_type_matches = ["OpenGear"]
            expect.expect(device_type_matches)
            device_type = OPENGEAR_OG
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            device_type = UNKNOWN

    if device_type == UNKNOWN:
        try:
            expect.send("cat /etc/build\r\n")
            device_type_matches = ["Operations Manager"]
            expect.expect(device_type_matches)
            device_type = OPENGEAR_NG
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            device_type = UNKNOWN

    return device_type

def get_factory_default(expect, device_type):
    """
    Determine whether a device is in its factory state (i.e. ready to be provisioned by us)
    """
    if device_type in (CISCO_IOS, CISCO_XE):
        expect.send("show startup-config\r\n")
        try:
            expect.expect("startup-config is not present")
            return True
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return False
    if device_type == CISCO_XR:
        expect.send("show configuration persistent\r\n")
        try:
            expect.expect("Last configuration change")
            return False
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return True
    if device_type == OPENGEAR_OG:
        expect.send("config -g config.interfaces.wan.mode\r\n")
        try:
            expect.expect("config.interfaces.wan.mode")
            return False
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return True
    if device_type == OPENGEAR_NG:
        # TODO unclear if we can determine whether NGCS is in a factory state
        return False
    return False

def get_mac_address(expect, device_type):
    mac = None
    if device_type == ARISTA:
        expect.send("show interface management 1\r\n")
        try:
            expect.expect(r"Hardware is \S+, address is \S+\.\S+\.\S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.decode('utf-8').split(' ')
                if len(split) > 5:
                    mac = split[5]
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return mac
    if device_type in (CISCO_IOS, CISCO_XE):
        expect.send("show version\r\n")
        try:
            # This gets the base MAC address, rather than the MAC address of a
            # particular port. All port MACs should be based off the base MAC address,
            # so can be matched by wildcard.
            expect.expect(r"[Bb]ase [Ee]thernet MAC Address\s*:\s+\S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.decode('utf-8').split(' ')
                mac = split[len(split) - 1]
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return mac
    if device_type == CISCO_XR:
        # This gets the first Management interface MAC address
        expect.send("show interfaces | include Hardware is Management\r\n")
        try:
            expect.expect(r"Hardware is .+, address is \S+\.\S+\.\S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.decode('utf-8').split(' ')
                mac = split[len(split) - 1]
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return mac
    if device_type == OPENGEAR_OG:
        expect.send("setfset -q 0\r\n")
        try:
            expect.expect(r"MAC address for \S+: \S+:\S+:\S+:\S+:\S+:\S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.decode('utf-8').split(' ')
                mac = split[len(split) - 1]
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return mac
    if device_type == OPENGEAR_NG:
        expect.send("ip addr show net1\r\n")
        try:
            expect.expect(r"link/ether \S+:\S+:\S+:\S+:\S+:\S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.decode('utf-8').split(' ')
                mac = split[len(split) - 3]
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return mac
    return mac

def get_serial_number(expect, device_type):
    serial = None
    if device_type == ARISTA:
        expect.send("show version\r\n")
        try:
            expect.expect(r"Serial number:\s*\S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.split()
                serial = split[len(split) - 1].decode('utf-8')
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return serial
    if device_type in (CISCO_IOS, CISCO_XE):
        expect.send("show version\r\n")
        try:
            expect.expect(r"System [Ss]erial [Nn]umber\s*:\s*\S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.decode('utf-8').split(' ')
                serial = split[len(split) - 1]
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return serial
    if device_type == OPENGEAR_OG:
        expect.send("setfset -q n\r\n")
        try:
            expect.expect(r"Serial number: \S+")
            match_group = expect.match.group()
            if match_group:
                split = match_group.decode('utf-8').split(' ')
                serial = split[len(split) - 1]
        except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
            return None
        return serial
    if device_type == OPENGEAR_NG:
        # TODO unclear on how to get serial number for NGCS
        return None
    return serial

def get_label_banner(expect):
    """
    Determine label from device banner. Note: getting to the banner requires sending an exit
    sequence which will kill any session/state. This function does NOT send that, and assumes
    it has already been sent
    """
    label = None

    banner_matches = [
        r'FreeBSD\S+ \(\S+',
        r'\S+ con[0-9]* is now available',
        r'\S+ R[S]*P0.* is now available',
        r'\S+ \(tty',
        r'\S+ login:'
    ]

    try:
        matched = expect.expect(banner_matches)
        if 0 <= matched <= len(banner_matches):
            if matched == 0:
                label = expect.match.group().decode("utf-8").split(' ')[1]
            else:
                label = expect.match.group().decode("utf-8").split(' ')[0]
            if label: # TODO validate label?
                return label
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        return None

    return None

def get_label_prompt(expect):
    """
    Determine label from device prompt (in either Prompt or Enabled state).

    This function assumes the device is in the correct state to read the prompt
    """
    label = None

    prompt_matches = [
        r'\S+>',
        r'\S+#'
    ]

    try:
        matched = expect.expect(prompt_matches)
        if matched == 0:
            label = expect.match.group().decode("utf-8").split('>')[0]
        elif matched == 1:
            label = expect.match.group().decode("utf-8").split('#')[0]
        else:
            return None
        return label
    except (pexpect.exceptions.TIMEOUT, pexpect.exceptions.EOF):
        return None
