import os
import json
import socketserver
from io import BytesIO
from . import handle
from http.server import HTTPServer, BaseHTTPRequestHandler


class ThreadedHTTPServer(socketserver.ThreadingMixIn, HTTPServer):
    """
    Pseudo-class designed purely to turn on threading for receiving HTTP
    requests.
    The purpose of this class is primarily to enable threading for the HTTP
    server daemon.
    """
    daemon_threads = True
    timeout = 0.01 #Timeout for request handling

class SimpleHTTPRequestHandler(BaseHTTPRequestHandler):
    """
    Basic HTTP request handler which respond to POST and GET requests
    for valid endpoints.
    """
    def __init__(self, job_queue, database, *args):
        self.job_queue = job_queue
        self.database = database
        BaseHTTPRequestHandler.__init__(self, *args)

    def do_GET(self):
        method = "GET"
        split_path = os.path.splitext(self.path)
        request_extension = split_path[1]
        if request_extension == "":
            handler = handle.TemplateHandler(self.database)
            if handler.check(self.path, method):
                handler.get_route(self.path, method)
            else:
                handler = handle.BadRequestHandler()
        else:
            handler = handle.BadRequestHandler()
        self.respond({
            'handler': handler
        })

    def do_POST(self):
        method = "POST"
        content_length = int(self.headers['Content-Length'])
        body = self.rfile.read(content_length)
        split_path = os.path.splitext(self.path)
        request_extension = split_path[1]
        if request_extension == "":
            handler = handle.TemplateHandler(self.database)
            if handler.check(self.path, method):
                data = body.decode("utf-8")
                try:
                    data = json.loads(data.replace("\"", "\'"))
                except json.decoder.JSONDecodeError:
                    # TODO should return empty list on bad message body
                    # - This will actually cause failures further down
                    #   since we expect a dict() later on
                    pass
                self.job_queue.put_nowait({self.path: data})
                self.send_response(200)
                self.end_headers()
                self.wfile.write(b"OK")
                return
        # self.send_header('Content-type', handler.content_type)
        self.send_response(404)
        self.end_headers()
        self.wfile.write(b"\n")

    def do_PUT(self):
        self.send_response(404)
        self.end_headers()
        response = BytesIO
        response.write(b"PUT not implemented")

    def handle_http(self, handler):
        if handler.status == 200:
            content = handler.get_route(self.path, "GET")
            if content == {}:
                handler.status = 404
                content = "404 Not Found"
        else:
            content = "404 Not Found"
        status_code = handler.status
        # self.send_header('Content-type', handler.content_type)
        self.send_response(status_code)
        self.end_headers()
        return bytes(str(content), 'UTF-8')

    def respond(self, opts):
        response = self.handle_http(opts['handler'])
        # Handle case of empty response?
        self.wfile.write(response)
