import json
import base64
import requests
from typing import List, Dict
from central_sdi import util
from netops.lighthouse import nodes, groups

requests.urllib3.disable_warnings()


class Smartgroup:
    """
    Represents a Lighthouse Smart Group object as retrieved by the LH REST API.
    """

    def __init__(self, node_data):
        self._id = node_data.get("id")
        self._name = node_data.get("name")
        self._query = node_data.get("query")
        self._rights = node_data.get("rights")

    @property
    def id(self):
        return self._id

    @property
    def name(self):
        return self._name

    @property
    def query(self):
        return self._query

    @property
    def rights(self):
        return self._rights


def _get_port_smartgroup(port_sg_id: str):
    ports_sg = util.make_request(f"https://{util.lh_addr()}/api/{util.get_api_version()}/ports/smartgroups/{port_sg_id}")
    return json.loads(ports_sg.json()["smartgroup"]["query"])


def _get_node_smartgroup(node_sg_id: str):
    nodes_sg = util.make_request(f"https://{util.lh_addr()}/api/{util.get_api_version()}/nodes/smartgroups/{node_sg_id}")
    return json.loads(nodes_sg.json()["smartgroup"]["query"])


def _search_with_query(query: Dict) -> Dict:
    """
    Search all managed devices using a query. Returns a dictionary-like object
    representing the json.
    """
    # Convert the query to base64
    encoded_query = base64.b64encode(json.dumps(query).encode("utf-8")).decode("utf-8")

    # Run a search for nodes using the constructed query
    params = f"jb64={encoded_query}"
    resp = util.make_request(f"https://{util.lh_addr()}/api/{util.get_api_version()}/search/managed?{params}")
    search_id = resp.json()["search"]["id"]

    # Get the result of the search, using the searchId from the previous step
    params = f"searchId={search_id}"
    resp = util.make_request(f"https://{util.lh_addr()}/api/{util.get_api_version()}/managed_devices?{params}")
    resp.raise_for_status()

    return resp.json()


def get_nodes_in_group(group: groups.Group) -> List[nodes.Node]:
    """
    This function returns all nodes which a group matches
    (WARNING: the group does not necessarily allow access to these nodes).

    Groups in Lighthouse have up to two smartgroups associated with them:
      - A Node Smartgroup (Smart Group)
      - A Port Smartgroup (Managed Device Filter)
    A group cannot have a Port Smartgroup without having a Node Smartgroup.

    A node is accessible if it matches the Node Smartgroup, and at least one
    port matches the Port Smartgroup. If there is no Port Smartgroup, only the
    Node Smartgroup has to match.

    This DOES NOT discriminate in ANY other way (e.g: only enrolled/unenrolled
    nodes, roles, permissions, disabled groups).
    """

    # Get the node smartgroup query and the port smartgroup query
    nodes_query = _get_node_smartgroup(group.smart_group)

    # Run a quick string modification on the node query, so it makes sense to the new endpoint
    _update_query_fieldname(nodes_query)

    # Join the node query and the port query
    combined_query = {
        "type": 1,  # "AND" condition
        "items": [nodes_query],
    }
    if group.managed_device_filter:
        combined_query["items"].append(_get_port_smartgroup(group.managed_device_filter))

    return [nodes.Node(n) for n in _search_with_query(combined_query)["nodes"]]


def _update_query_fieldname(query):
    """
    Recursively update fieldnames within a query to have "node:" before them.
    """
    if "fieldname" in query and query["fieldname"].startswith(("config:", "tag:", "status:")):
        query["fieldname"] = "node:" + query["fieldname"]

    if "items" in query:
        for item in query["items"]:
            _update_query_fieldname(item)
