"""
router is the entrypoint into actual api logic
"""

import falcon
import falcon.status_codes as http_status
from typing import (
    Any,
    Callable,
    Dict,
    List,
    Optional,
)
from netops.lighthouse import (
    nodes as lh_nodes,
)
from netops import (
    automation_gateway as ag,
)
from netops.automation_gateway import (
    job,
)
from central_ag.api import (
    auth,
    authz,
    auth_token,
    devices,
    devices_names,
    devices_redirects,
    manual_nmap,
    nodes_status,
)


class Root:
    """
    Root wires the api business logic up
    """

    def __init__(
            self,
            *,
            identity: Optional[auth.Identity] = None,
            ag_auth: Optional[ag.AuthStub] = None,
            ag_devices: Optional[ag.DevicesStub] = None,
            ag_nmap_scan: Optional[job.NmapScanStub] = None,
            get_nodes: Optional[Callable[[str, List[str]], Dict[str, lh_nodes.Node]]] = None,  # (token, node_ids)
    ):
        """
        constructs a new root
        """
        self._routes = {
            '/auth_tokens': auth_token.Tokens(
                identity=identity,
                ag_auth=ag_auth,
                get_nodes=get_nodes,
            ),
            '/auth_tokens/validate': auth_token.Validate(
                identity=identity,
                ag_auth=ag_auth,
            ),
            '/devices': devices.Devices(
                identity=identity,
                ag_devices=ag_devices,
                get_nodes=get_nodes,
            ),
            '/devices/{device_id}': devices.Device(
                identity=identity,
                ag_devices=ag_devices,
                get_nodes=get_nodes,
            ),
            '/devices_names': devices_names.DevicesNames(
                ag_devices=ag_devices,
            ),
            '/devices_redirects': devices_redirects.DevicesRedirects(
                ag_devices=ag_devices,
            ),
            '/manual_nmap_scan': manual_nmap.ManualNmap(
                ag_devices=ag_nmap_scan,
            ),
            '/routes': Routes(),
            '/nodes/{node_id}/status': nodes_status.Resource(
                get_nodes=get_nodes,
            ),
        }

    @property
    def routes(self) -> Dict[str, Any]:
        """
        routes describes the falcon api resources keyed on their path
        """
        return self._routes

    def register(self, app: falcon.App):
        """
        register sets up a falcon app with the routes provided by this instance
        """
        for path, resource in self._routes.items():
            app.add_route(path, resource)


class Routes:
    def on_get(self, request, response):
        """Routes GET endpoint.
        ---
        description: Get list of AG UI routes
        responses:
            200:
                description: UI routes
        """
        token = request.context.get("token")

        response.status = http_status.HTTP_200
        routes = {"Automation Gateway": []}

        if authz.can(token, "/nom/ag/devices", "get"):
            routes["Automation Gateway"].append({
                "name": "Devices",
                "route": "devices"
            })

        response.media = routes
