"""
This module provides REST handlers for Device Redirects
"""
import falcon.status_codes as http_status
from central_ag import devices_redirects
from central_ag.api import (
    schema,
    authz,
)
from netops import automation_gateway as ag
from typing import Optional


class DevicesRedirects:
    """
    Handlers for updating and reading devices' customized redirects.
    """
    schema = schema.DeviceRedirectList()

    def __init__(
            self,
            *,
            ag_devices: Optional[ag.DevicesStub] = None,
    ):
        self._ag_devices = ag_devices

    def on_get(self, request, response):
        """
        Devices' names GET endpoint.
        ---
        description: Retrieve all of the devices' custom redirects, and return them as a list.
        responses:
            200:
                description: Successfully retrieved
                content:
                    application/json:
                        schema: DeviceRedirectList
        """
        if not authz.can(request.context.get("token"), "/nom/ag/devices_names", "get"):
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
            return

        request.context.result = devices_redirects.load()

    def on_put(self, request, response):
        """
        Devices' custom redirects PUT endpoint.
        ---
        description: Save or update the redirect configuration of devices.
        requestBody:
            content:
                application/json:
                    schema: DeviceRedirectList
        responses:
            204:
                description: Successfully updated
            400:
                description: Invalid request
            403:
                description: Permission denied
        """
        # Validate the request
        mac_list = set()
        token = request.context.get("token")
        for device in request.context.json["devices_redirects"]:
            mac = device.get("mac", None)
            redirect = device.get("redirect", None)
            route = device.get("route", None)
            if mac is None or redirect is None or route is None:
                response.status = http_status.HTTP_400
                response.media = {"error": "Invalid request - Mac, Redirect and Route are all required in request"}
                return

            if mac in mac_list:
                # Check if setting different names for the same device
                response.status = http_status.HTTP_400
                response.media = {"error": f"Invalid request - Duplicated records defined for device: '{mac}'"}
                return
            # Add to the groups list to check if the group has been defined before
            mac_list.add(mac)
        if authz.can(token, "/nom/ag/devices_names", "put"):
            devices_redirects.store(request.context.json)
            response.status = http_status.HTTP_204
        else:
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
            return
