#!/usr/bin/python3
"""
provd responds to events on particular data store keys, and runs scripts
according to it's configuration.

The configuration file by default lives in /etc/provd.conf but can be specified
by passing in the --config-file=/file/path flag. The directory where the scripts
can be found must be provided by the --script-dir=/path/ argument.

The scripts can be jinja2 templates, and will be templated only if they end in
the .j2 suffix.

Each of the scripts will be run in a new thread for the specified timeout
limit, which is 60 seconds by default.
"""
import sys
import os
import signal
import json
import redis
import time
import getopt
import re
import tempfile
import stat

from jinja2 import Environment, FileSystemLoader
from jinja2.exceptions import TemplateNotFound

from . import prov_config, prov_manager, target_device
from .prov_manager import sprint, eprint

# handle for the redis connection
redis_port = 6379
if 'REDIS_PORT' in os.environ:
    redis_port = os.environ['REDIS_PORT']

redis_handle = redis.StrictRedis(host="localhost", port=redis_port, db=0)
redis_channel = "dop"

""" globals for the provd configuration """
running_config = None
config_file = "/etc/provd.conf"

""" prov_man tracks the currently running scripts managed by provd """
prov_man = prov_manager.ProvManager()

""" jinja2_env is the templating handle """
jinja2_env = None


def process_message(config, td):
    """
    process_message receives events of TargetDevice keys from
    the datastore. This indicates that the device config has been updated, and
    may need to have the provisioning script run.

    @config is the running configuration for provd.

    @td is a TargetDevice object as parsed from the datastore.
    """

    # Compare the resource bundles of the TD against the required post-provisioning
    # parameters for that resource bundle. If all parameters are satisfied, run
    # the associated script.
    changed = False
    for name in td.bundles:
        resource = config.get_resource_by_name(name)
        if not resource:
            continue

        # If no files are specified for the resource bundle, this script will not be run.
        # TODO this behaviour may need to be revisited in the context of bundles which intentionally
        # specify no files, such as cloud provisioning and Opengear enrollment
        if not resource.files:
            continue

        # Check all files have been downloaded by the td
        matched = [file for file in td.files_downloaded if (
            file in resource.files)]
        if sorted(matched) == sorted(resource.files):
            # Set the provisioned flag if not already set
            if not td.provisioned:
                td.provisioned = True
                changed = True

            if resource.script != "" and resource.script not in td.post_provision_scripts_run:
                # Template the script for the target device before running it.
                # It's possible the data will change, so we want this to happen as
                # late as possible.
                script = template_script(td, resource, jinja2_env)
                run_script(td, resource, script, resource.script)
                td.add_post_provision_script_run(resource.script)
                changed = True

    return changed


def template_script(td, resource, jinja_handle):
    """
    template_script replaces and of the TargetDevice variables in the script
    with the device's data that is about to be run. This function returns the
    path to the templated script which is a copy of the original script with
    the templated data replaced OR the original script if no data was to be
    templated. The caller needs to delete the copy of the script.
    """
    fd, new_file = tempfile.mkstemp()
    with os.fdopen(fd, "w") as f:
        if resource.script.endswith(".j2"):
            try:
                template = jinja_handle.get_template(
                    os.path.basename(resource.script))
                rendered_template = template.render(
                    nom_device_ipv4_address=td.ip,
                    nom_device_mac_address=td.mac,
                    nom_device_hostname=td.hostname)
                f.write(rendered_template)
            except TemplateNotFound as err:
                eprint("error templating script: {}".format(str(err)))
        else:
            with open(resource.script, "r") as file:
                f.write(file.read())
    # 0775 permission
    os.chmod(new_file, stat.S_IRWXG | stat.S_IRWXU |
             stat.S_IXOTH | stat.S_IROTH)
    return new_file


def run_script(td, resource, tmp_script, actual_script):
    """
    run_script executes the script in a separate thread. The script
    execution will be cancelled after a timeout configured
    in the resource itself.

    @resource is a Resource object

    @script is the script file to be run
    """
    sprint("running post-provisioning script {} for host {} with timeout {} sec"
        .format(actual_script, td.mac, resource.timeout))
    sys.stdout.flush()

    # Small sleep to make sure the "running" log message appears after
    # the stdout of the script, since it has to go through more processing
    time.sleep(2)
    prov_man.add_proc(prov_manager.Proc(
        tmp_script=tmp_script, actual_script=actual_script, timeout=resource.timeout, mac=td.mac))


def load_config(file=config_file):  # pylint: disable=inconsistent-return-statements
    """
    load_config reads the configuration from @file and returns the json object.
    """
    config_json = None
    try:
        with open(file) as f:
            config_json = f.read()
    except IOError as err:
        eprint(str(err))
        cleanup(1)

    if not config_json:
        eprint("empty configuration file provided")
        cleanup(1)

    # parse into known config structure
    try:
        return prov_config.from_json(config_json)
    except json.decoder.JSONDecodeError as err:
        eprint("error decoding json configuration file: {}".format(err.msg))
        cleanup(1)
    except prov_config.InvalidResource as err:
        eprint("error parsing configuration for provd: {}".format(err.message()))
        cleanup(1)


def sighup_handler(signum, frame): # pylint: disable=unused-argument
    """
    Reload the config file on event of a sighup. Keep any active threads running.
    """
    global running_config
    eprint("received HUP, reloading configuration file")
    running_config = load_config(config_file)


signal.signal(signal.SIGHUP, sighup_handler)


def sigterm_handler(signum, frame): # pylint: disable=unused-argument
    """
    Close the redis handle, and wait for any running threads to finish before
    exiting the program.
    """
    cleanup(0)


signal.signal(signal.SIGTERM, sigterm_handler)
signal.signal(signal.SIGINT, sigterm_handler)


def cleanup(status):
    """
    cleanup closes the redis connect, waits for any running threads to complete,
    and exits with the given status code.
    """
    sprint("closing redis connection")
    redis_handle.connection_pool.disconnect()
    sprint("connection closed")

    sprint("waiting for any running scripts")
    while prov_man.running_count > 0:
        sprint("{} script(s) still running".format(prov_man.running_count))
        time.sleep(1)

    prov_man.stop()
    sprint("all scripts completed, exiting")
    sprint("exiting")
    sys.exit(status)


def main():
    """
    configure the provd daemon, connect to redis, and start listening for
    incoming messages. Each message constructs a TargetDevice and passes it
    down to the event handler.
    """
    global running_config
    global config_file

    try:
        optlist, _ = getopt.getopt(
            sys.argv[1:], "", ["config-file=", "script-dir="])
    except getopt.GetoptError as err:
        print((str(err)))
        sys.exit(2)

    script_dir = None
    for o, a in optlist:
        if o == "--config-file":
            config_file = a
        if o == "--script-dir":
            script_dir = a

    if not script_dir:
        eprint("you must specify a script directory")
        sys.exit(2)

    # name_r is the check for TargetDevice specific keys.
    name_r = "{}:.*".format(target_device.KEY_PREFIX)
    name_re = re.compile(name_r)

    # load the initial config
    running_config = load_config(config_file)

    # prepare the jinja2 templating engine
    global jinja2_env
    jinja2_env = Environment(loader=FileSystemLoader(script_dir))

    sprint("connecting to redis")
    p = redis_handle.pubsub(ignore_subscribe_messages=True)
    p.subscribe("__keyevent@0__:set")  # subscribes to all key set events.
    sprint("subscribed to all key set events, waiting for messages")

    sprint("running process manager in the background")
    prov_man.start()

    for message in p.listen():
        # we only care about TargetDevice key sets
        key = message['data'].decode("utf-8", "ignore")
        if name_re.match(key):
            # get the data matching this key and process any changes
            try:
                td = target_device.TargetDevice(redis_handle).from_json(
                    redis_handle.get(key).decode("utf-8", "ignore"))
                changed = process_message(running_config, td)
                if changed:
                    td.save()
            except target_device.LoadError as err:
                eprint("error loading TargetDevice from data: {}".format(err.message))


if __name__ == "__main__":
    main()
