"""
prov_config provides the Config and Resource classes for the provd program.
These classes define the device resource bundles and the associated
files / scripts.
"""
import json
from json import JSONEncoder


class Error(Exception):
    """Base class for exceptions in the prov_config module."""
    def __init__(self, msg): # pylint: disable=super-init-not-called
        self._message = msg

    def message(self):
        """ returns the exceptions error message """
        return self._message


class InvalidResource(Error):
    """ Raised when an invalid configuration has been observed """


class ConfigDecoder(JSONEncoder):
    """ Config specific json decoder to create a class we can work with """

    def default(self, o):
        out = {
            'resources': []
        }

        for resource in o.resources:
            res = {
                'name': resource.name,
                'files': resource.files,
                'script': resource.script,
                'timeout': resource.timeout,
                'provision_after': resource.provisionAfter
            }
            out['resources'].append(res)

        return out


class Config:
    """
    Config represents a prov_config daemon configuration. We use this to
    construct the daemon configuration file instead of manually building the
    json object.
    """
    def __init__(self, resources=None):
        if resources is None:
            self._resources = []
        else:
            self._resources = resources

    @property
    def resources(self):
        """ returns the resources specified to the configuration """
        return self._resources

    def add_resource(self, resource):
        """ append a new resource to the configuration """
        self.resources.append(resource)

    def get_resource_by_name(self, name):
        """return the resource with the given name"""
        match = [resource for resource in self.resources if resource.name == name]
        if len(match) == 1:
            return match[0]
        return None

    def to_json(self):
        """
        converts the Config object to a json string
        """
        return json.dumps(self, cls=ConfigDecoder)


class Resource:
    """
    Resource represents a component of a prov_config configuration file.
    Specifically, a resource basically represents a script and prerequisite
    list of files before the script is executed.
    """
    def __init__(self, name, script, files=None, timeout=900, provisionAfter=None):
        self._name = name
        self._script = script

        if files is None:
            self._files = []
        else:
            self._files = files

        if timeout:
            self._timeout = timeout
        else:
            self._timeout = 900

        if provisionAfter is None:
            self._provisionAfter = []
        else:
            self._provisionAfter = provisionAfter

    @property
    def name(self):
        """ return the name of the resource configuration """
        return self._name

    @property
    def files(self):
        """ return the list of all files associated with the configuration """
        return self._files

    @files.setter
    def files(self, value):
        """ replace the current list of files with a new list """
        self._files = value

    @property
    def timeout(self):
        """ the time the script can run for before being killed """
        return self._timeout

    @timeout.setter
    def timeout(self, value):
        """ override the default timeout for scripts """
        self._timeout = value

    @property
    def script(self):
        """ specify a script to associate with the resource """
        return self._script

    @property
    def provisionAfter(self):
        """
        A list of other resources that must be provisioned prior to this
        particular resource before the script will be executed.
        """
        return self._provisionAfter

    @provisionAfter.setter
    def provisionAfter(self, value):
        """
        replace the existing list of resources to wait for with a new list
        """
        self._provisionAfter = value


def from_json(json_str):
    """
    from_json returns a Config object with associated resources from a json
    blob.
    @json_str should be a (non-serialized) json string that can be used in a call to
    json.loads().
    """
    serialized = json.loads(json_str)
    resources = []
    for resource in serialized['resources']:
        res = Resource(name=resource['name'], script=resource['script'])
        if 'timeout' in resource:
            timeout = resource['timeout']
            if not isinstance(timeout, int) or timeout <= 0:
                raise InvalidResource("invalid timeout provided")

            res.timeout = resource['timeout']
        if 'files' in resource:
            res.files = resource['files']
        if 'provision_after' in resource:
            res.provisionAfter = resource['provision_after']

        resources.append(res)

    resource_names = [resource.name for resource in resources]

    for resource in resources:
        for dependency in resource.provisionAfter:
            if dependency not in resource_names:
                raise InvalidResource("invalid dependency: resource bundle {} does not exist".format(dependency))

    return Config(resources)
