"""
client implements models a client for interaction with the ngcs api
"""
from typing import Callable, Dict, List, Optional, Tuple
from urllib import parse
import requests

from ..ngcs import logic, models

# _timeout is the default http request timeout used by the Api class
_timeout = 60.0


class Api:
    """
    implements a client for the ngcs api
    """

    def __init__(
            self,
            address: str,
            cert: Tuple[str, str],
            version: str = '2.2',
            verify: bool = False,
            request: Callable = requests.request,
    ):
        self._address = address
        self._cert = cert
        self._version = version
        self._verify = verify
        self._request = request

    def _url(self, path: str, query: Optional[Dict[str, List[str]]] = None):
        """
        url builds a fully qualified api url from a relative api path, e.g. "firewall/services"

        :param path:
        :param query:
        :return:
        """
        return parse.urlunsplit((
            'https',
            self._address,
            "/".join(map(lambda x: str(x).rstrip('/'), ('api', 'v{}'.format(self._version), path))),
            parse.urlencode(query, doseq=True) if query is not None else '',
            '',
        ))

    def get_firewall_services(self) -> List[models.FirewallService]:
        response = self._request(
            method='GET',
            url=self._url('firewall/services'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return [models.FirewallService(data) for data in response.json()['firewall_services']]

    def post_firewall_service(self, service: models.FirewallService) -> models.FirewallService:
        response = self._request(
            method='POST',
            url=self._url('firewall/services'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
            json={'firewall_service': service.data},
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return models.FirewallService(response.json())

    def put_firewall_service(self, service: models.FirewallService) -> models.FirewallService:
        response = self._request(
            method='PUT',
            url=self._url('firewall/services/{}'.format(service.id)),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
            json={'firewall_service': service.data},
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return models.FirewallService(response.json())

    def update_firewall_service(self, service: models.FirewallService) -> models.FirewallService:
        """
        updates the given service with the given name in the config, performing a POST or
        PUT if necessary and as appropriate, it's not atomic

        WARNING for some god-forsaken reason both name AND label must be unique, so it may
                die screaming if the label is not unique

        :param service:
        :return:
        """
        try:
            data = service.data
            service = logic.find_firewall_service(self.get_firewall_services(), name=service.name)
            data['id'] = service.id
            if data == service.data:
                return service
            return self.put_firewall_service(models.FirewallService(data))
        except logic.NotFoundException:
            return self.post_firewall_service(service)

    def get_firewall_policies(self) -> List[models.FirewallPolicy]:
        response = self._request(
            method='GET',
            url=self._url('firewall/policies'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return [models.FirewallPolicy(data) for data in response.json()['firewall_policies']]

    def get_firewall_zones(self) -> List[models.FirewallZone]:
        response = self._request(
            method='GET',
            url=self._url('firewall/zones'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return [models.FirewallZone(data) for data in response.json()['firewall_zones']]

    def get_firewall_rules(self) -> List[models.FirewallRule]:
        response = self._request(
            method='GET',
            url=self._url('firewall/rules'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return [models.FirewallRule(data) for data in response.json()['firewall_rules']]

    def post_firewall_rule(self, rule: models.FirewallRule) -> models.FirewallRule:
        response = self._request(
            method='POST',
            url=self._url('firewall/rules'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
            json={'firewall_rule': rule.data},
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return models.FirewallRule(response.json())

    def get_interfaces(self) -> List[models.Interface]:
        response = self._request(
            method='GET',
            url=self._url('physifs'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return [models.Interface(data) for data in response.json()['physifs']]

    def get_connections(self) -> List[models.Connection]:
        response = self._request(
            method='GET',
            url=self._url('conns'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return [models.Connection(data) for data in response.json()['conns']]

    def post_connection(self, service: models.Connection) -> models.Connection:
        response = self._request(
            method='POST',
            url=self._url('conns'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
            json={'conn': service.data},
        )
        response.raise_for_status()
        if response.status_code != 200:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return models.Connection(response.json()['conn'])

    def get_failover_cellular(self) -> models.FailoverToCellular:
        response = self._request(
            method='GET',
            url=self._url('ip_passthrough'),
            verify=self._verify,
            cert=self._cert,
            timeout=_timeout,
        )
        # old NGCS does not support this API
        if response.status_code == 404:
            data = {
                    "enabled":False,
                    "service_intercepts":{"ssh": None, "https": None},
                    "mac_address":"",
                    "passthrough_physif":""}
        elif response.status_code == 200:
            data = response.json()['ip_passthrough']
        else:
            raise ValueError('unexpected http status: {}'.format(response.status_code))
        return models.FailoverToCellular(data)
