import functools
import asyncio
import contextlib
import threading
from datetime import (
    datetime,
    timezone,
    timedelta,
)
from typing import (
    Union,
    Optional,
)


@contextlib.asynccontextmanager
async def ticker(
        rate: Union[timedelta, float, int],
        *,
        count: int = -1,
        initial: bool = False,
        loop: Optional[asyncio.AbstractEventLoop] = None,
):
    """async context manager that provides an async generator, which will tick
    (yield the datetime at the time when the tick was scheduled) every rate,
    until either the context manager exits, or the count limit (indicated by a
    count > 0) is reached, note that the initial flag may be set to schedule
    a tick (soon) without an initial wait"""
    if not isinstance(rate, timedelta) and isinstance(rate, (float, int)):
        rate = timedelta(seconds=rate)
    if not isinstance(rate, timedelta) or rate <= timedelta():
        raise ValueError('aioutil.ticker rate must be a timedelta > 0')
    if not isinstance(count, int) or count == 0:
        raise ValueError('aioutil.ticker count must be an int < 0 (no limit) or > 0 (limited)')

    if count > 0 and not initial:
        count += 1

    if loop is None:
        loop = asyncio.get_event_loop()

    future = asyncio.ensure_future(loop.create_future(), loop=loop)

    async def generate():
        nonlocal count, initial
        while not future.done() and count != 0:
            if count > 0:
                count -= 1
            tick = datetime.now(timezone.utc)
            if initial:
                yield tick
            initial = True
            await asyncio.wait({future}, timeout=(rate - (datetime.now(timezone.utc) - tick)).total_seconds(), loop=loop)  # pylint: disable=deprecated-argument

    try:
        yield generate()
    finally:
        try:
            future.set_result(None)
        except asyncio.InvalidStateError:
            pass


class DoneError(Exception):
    """raised if the future provided to Done.wait didn't result before done"""


class Done:
    """signaling mechanism, providing an awaitable "done" flag, with very nice
    cancelation support, and a degree of thread safety"""

    def __init__(self, *, loop: Optional[asyncio.AbstractEventLoop] = None):
        self._ready = False
        self._loop = loop  # possibly initialised lazily
        self._future = None  # initialised lazily
        self._lock = threading.Lock()
        self._value = False

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()

    async def __aenter__(self):
        return self.__enter__()

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        return self.__exit__(exc_type, exc_val, exc_tb)

    def __await__(self):
        """WARNING do not await outside of the event loop this is is for"""
        return self.future.__await__()

    def __bool__(self):
        with self._lock:
            return self._value

    @property
    def loop(self) -> asyncio.AbstractEventLoop:
        with self._lock:
            self._lazy()
            return self._loop

    @property
    def future(self):
        with self._lock:
            self._lazy()
            return self._future

    def close(self):
        """marks as done, a valid operation regardless of the state, note that
        any change to the future will propagate asynchronously"""
        with self._lock:
            if not self._value:
                if self._ready:
                    self._call_close()
                self._value = True

    @asyncio.coroutine
    def wait(self, fut):
        """asyncio.wait_for except the cancelation trigger is close and
        DoneError may be raised rather than asyncio.TimeoutError"""
        future = self.future
        loop = future.get_loop()

        waiter = loop.create_future()
        release_waiter = functools.partial(_release_waiter, waiter)

        future.add_done_callback(release_waiter)

        try:
            fut = asyncio.ensure_future(fut, loop=loop)
            fut.add_done_callback(release_waiter)

            # wait until the future completes or done
            try:
                yield from waiter
            except asyncio.CancelledError:
                fut.remove_done_callback(release_waiter)
                fut.cancel()
                raise

            if fut.done():
                return fut.result()

            fut.remove_done_callback(release_waiter)
            fut.cancel()
            raise DoneError()
        finally:
            future.remove_done_callback(release_waiter)

    def _lazy(self):
        """lazy init of loop and future"""
        if not self._ready:
            if self._loop is None:
                self._loop = asyncio.get_event_loop()
            self._future = self._loop.create_future()
            if self._value:
                self._call_close()
            self._ready = True

    def _call_close(self):
        try:
            self._loop.call_soon_threadsafe(self._close_sync)
        except RuntimeError:
            # handles error like:
            # File "/usr/lib/python3.8/asyncio/base_events.py", line 508, in _check_closed
            #   raise RuntimeError('Event loop is closed')
            self._close()

    def _close_sync(self):
        with self._lock:
            self._close()

    def _close(self):
        try:
            self._future.set_result(True)
        except asyncio.InvalidStateError:
            pass


def _release_waiter(waiter, *args):  # pylint: disable=unused-argument
    if not waiter.done():
        waiter.set_result(None)
