from marshmallow import Schema, fields, validate


### Cert Schemas ###

class CertConfig(Schema):
    client_cert_days = fields.Int(
        required=True, validate=validate.Range(min=1, max=3650))
    client_cert_size = fields.Int(
        required=True, validate=validate.Range(min=1024))
    client_cert_default_cn = fields.String(required=True)


class CertRequest(Schema):
    name = fields.String(required=True)
    common_name = fields.String(required=False,
                                validate=validate.Regexp("^[^,]*$"))


class CertResponse(Schema):
    name = fields.String()
    serial = fields.Int()
    issuer = fields.String()
    subject = fields.String()
    fingerprint = fields.String()
    not_valid_before = fields.String()
    not_valid_after = fields.String()
    revoked = fields.Bool()


class CertList(Schema):
    certs = fields.List(fields.Nested(CertResponse))


### Network Access Policies ###

class Policy(Schema):
    """
    Policy models a network access policy.

    WARNING this schema is used directly to serialize and deserialize data,
            and as such care must be taken to avoid breaking existing user
            configuration data (in the storage api), between versions.
    """
    groups = fields.List(fields.String(), required=True, default=[])
    zones = fields.List(fields.String(), required=True, default=[])


class PolicyList(Schema):
    """PolicyList models a collection of Policy, please read it's warning."""
    policies = fields.List(fields.Nested(Policy), required=True, default=[])


### IPAccess Schemas ###

class IPAccessConfig(Schema):
    disable = fields.Bool()
    disable_routes = fields.Bool()
    route_metric = fields.Int(validate=validate.Range(min=0, max=9999))
    compress = fields.Bool()
    enable_wan = fields.Bool()
    enable_network_access_policies = fields.Bool()


### IPAccess Node Schemas ###

class IPAccessNode(Schema):
    node_id = fields.String()
    disabled = fields.Bool()


class IPAccessNodeList(Schema):
    nodes = fields.List(fields.Nested(IPAccessNode))


class IPAccessNodeAction(Schema):
    action = fields.String(validate=validate.OneOf(['enable', 'disable']))
    node_ids = fields.List(fields.String())


### Status Schemas ###

class ClientConnection(Schema):
    node_id = fields.String()
    node_name = fields.String()
    username = fields.String()
    common_name = fields.String()
    real_address = fields.String()
    virtual_address = fields.String()
    bytes_received = fields.Int()
    bytes_sent = fields.Int()
    connected_since = fields.String()
    client_id = fields.Int()


class Status(Schema):
    status = fields.List(fields.Nested(ClientConnection))


# All the schemas are enumerated here so they can be added to the API Spec
schemas = {
    'CertRequest': CertRequest,
    'CertResponse': CertResponse,
    'CertConfig': CertConfig,
    'ClientConnection': ClientConnection,
    'Policy': Policy,
    'PolicyList': PolicyList,
    'Status': Status,
    'IPAccessConfig': IPAccessConfig,
    'IPAccessNode': IPAccessNode,
    'IPAccessNodeList': IPAccessNodeList,
    'IPAccessNodeAction': IPAccessNodeAction,
}
