"""
manual_nmap implements the user-facing manual refreshing (stop current scan and restart) of nmap scans on attached nodes
"""
import falcon.status_codes as http_status
from typing import (
    Optional,
)
from netops.automation_gateway import (
    job,
)
from central_ag.api import (
    authz,
)
from google.protobuf import (
    json_format as jsonpb,
)


class ManualNmap:
    """
    Handlers for triggering and status-checking manual nmap scans
    """

    def __init__(
            self,
            *,
            ag_devices: Optional[job.NmapScanStub] = None,
    ):
        self._ag_devices = ag_devices

    def on_get(self, request, response):
        """
        GET Manual Nmap Scan Status
        ---
        description: Get the status of the current state of the manually triggered scan
        responses:
            200:
                description: Successfully retrieved status of scan
                content:
                    application/json:
                        schema: NmapScanGetJobResponse
            403:
                description: Permission denied
        """
        token = request.context.get("token")
        if authz.can(token, "/nom/ag/nmap_scan", "get"):
            response.status = http_status.HTTP_200
            response.media = jsonpb.MessageToDict(
                self._ag_devices.GetJob(job.NmapScanGetJobRequest()),
                including_default_value_fields=True,
                preserving_proto_field_name=True,
            )
        else:
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}

    def on_put(self, request, response):
        """
        PUT Manual Nmap Scan Trigger
        ---
        description: End any currently running node nmap scans and trigger a new scan on all nodes. Returns the status
                     of the newly started scans
        responses:
            200:
                description: Successfully triggered new scan
                content:
                    application/json:
                        schema: NmapScanDeleteJobResponse
            403:
                description: Permission denied
        """
        token = request.context.get("token")
        if authz.can(token, "/nom/ag/nmap_scan", "put"):
            response.status = http_status.HTTP_200
            response.media = jsonpb.MessageToDict(
                self._ag_devices.DeleteJob(job.NmapScanDeleteJobRequest()),
                including_default_value_fields=True,
                preserving_proto_field_name=True,
            )
        else:
            response.status = http_status.HTTP_403
            response.media = {"error": "Permission denied"}
