#!/usr/bin/env bash

# will conditionally replace the local server config with config from different (arbitrary) directories, moving the
# files since it's an atomic operation on linux, where the different directories are from environment variables
# provided via the args, it will only commence the update if the new config is deemed to be valid, and there is no
# intersection between the old and new file paths the old configuration is determined via the actual environment
# variables that this script is run with

set -o pipefail

# NOTE: avoid collisions with any of the other server_ scripts
EC_VALIDATION=3

die() { echo FATAL "$@" >&2; exit 1; }

server_dirs.sh \
|| exit "$?"

old_files="$( server_ensure.sh )"
case "$?" in
    0|3)
        # continue
        ;;
    *)
        die "refused to update local server: unknown destination error"
        ;;
esac

new_files="$( ( export "$@" && server_ensure.sh ) )"
case "$?" in
    0)
        # continue
        ;;
    "$EC_VALIDATION")
        echo "ERROR refused to update local server: invalid source config" >&2
        exit "$EC_VALIDATION"
        ;;
    *)
        die "refused to update local server: unknown source error"
        ;;
esac

combined_files_count="$( {
cat << EOF
$old_files
$new_files
EOF
} | wc -l )" \
&& [ "$combined_files_count" -gt 0 ] \
&& distinct_files_count="$( {
cat << EOF
$old_files
$new_files
EOF
} | sort | uniq | wc -l )" \
&& [ "$combined_files_count" -eq "$distinct_files_count" ] \
|| die "refused to update local server: source and destination are not distinct sets"

readarray -t old_files <<< "$old_files" \
&& readarray -t new_files <<< "$new_files" \
&& [ "${#old_files[@]}" -gt 0 ] \
&& [ "${#old_files[@]}" -eq "${#new_files[@]}" ] \
|| die "refused to update local server: failed to load file arrays"

# last validation, before actually updating the target
# may actually modify the target (old_files paths) but it's gotta create missing dirs sooo
for i in "${!old_files[@]}"; do
    old_file="${old_files[$i]}" \
    && new_file="${new_files[$i]}" \
    && [ ! -z "$old_file" ] \
    && [ ! -z "$new_file" ] \
    && [ -f "$new_file" ] \
    || die "refused to update local server: invalid files ($i, $old_file, $new_file)"

    old_dir="$( dirname -- "$old_file" )" \
    && [ ! -z "$old_dir" ] \
    && mkdir -p "$old_dir" \
    || die "refused to update local server: invalid target ($i, $old_file, $new_file)"
done

# the actual update, note this will generate backups, and there is no cleanup mechanism
for i in "${!old_files[@]}"; do
    old_file="${old_files[$i]}" \
    && new_file="${new_files[$i]}" \
    && mv --force --backup=numbered "$new_file" "$old_file" \
    || die "failed to update local server: exit code $? for ($i, $old_file, $new_file)"
done

# extract "$CA_DIR/certs.tar.gz", nuking any existing directory (backups are as the archive)
rm -rf -- "$CA_DIR/certs" \
&& tar --no-same-owner -C "$CA_DIR" -xzf "$CA_DIR/certs.tar.gz" certs \
&& chmod -R 644 -- "$CA_DIR/certs" \
&& chmod 755 -- "$CA_DIR/certs" \
|| die "failed to extract certs archive: $CA_DIR/certs.tar.gz"

# ensure the state is still sane and apply permission fixes
exec -- server_ensure.sh >/dev/null
