#!/usr/bin/env bash

# attempt to sync server config from the storage api
#
# notable exit cases:
#
# 1. something is broken with connecting to the storage api (failure)
# 2. storage api was checked successfully but none of the expected paths exist (failure(4), first run)
# 3. storage api was checked successfully but only some of the expected paths exist (failure(3), borked)
# 4. error with sync from the storage api to temp dirs (failure)
# 5. new files validation error or something went all _funky_, updating the server config from the temp files (failure)
# 6. everything might be fine (success)

set -o pipefail

# NOTE: avoid collisions with any of the other server_ scripts
EC_VALIDATION=3
EC_NOT_FOUND=4

die() { echo FATAL "$@" >&2; exit 1; }

num_existing=0
num_missing=0
for check_path in ca/index.json vpn/index.json; do
    check_code="$( timeout 10 curl -s -o /dev/null -w "%{http_code}" "$STORAGE_API_ADDR/$check_path" )" \
    || die "failed to check sdi-storage-api"
    case "$check_code" in
        200)
            ((num_existing++))
            ;;
        404)
            ((num_missing++))
            ;;
        *)
            die "unexpected http code $check_code from sdi-storage-api"
            ;;
    esac
done
if [ "$num_existing" -eq 0 ]; then
    exit "$EC_NOT_FOUND"
fi
if [ "$num_missing" -ne 0 ]; then
    echo "ERROR server configuration partially missing from sdi-storage-api" >&2
    exit "$EC_VALIDATION"
fi

ca_dir="$( mktemp -d )" \
&& vpn_dir="$( mktemp -d )" \
|| die "failed to create temp dirs"

function cleanup {
    exit_code="$?"
    trap - EXIT
    rm -rf "$ca_dir" "$vpn_dir"
    exit "$exit_code"
}
trap cleanup EXIT HUP INT QUIT PIPE TERM

rclone -q sync --http-url "$STORAGE_API_ADDR/ca/" :http: "$ca_dir" \
&& rclone -q sync --http-url "$STORAGE_API_ADDR/vpn/" :http: "$vpn_dir" \
|| die "failed to sync sdi-storage-api"

# validate, copy, validate, fix permissions
server_update.sh CA_DIR="$ca_dir" VPN_DIR="$vpn_dir"
