#!/usr/bin/env bash

# validates and normalises the state of the system / server level config e.g. the certificate authority and the server
# cert, and will print a list of relevant files (prefixed with the directory path from the relevant environment
# variable, such as CA_DIR or VPN_DIR, which are validated as not empty) to stdout, one per line, note it will exit
# with code 3 if the error was a validation error (it will still print a complete list of all files, but they may not
# all exist). Note that the directories defined as environment variables MUST exist for the "validation error" case to
# occur (exit code 3).
#
# this script is configured via environment variables

set -o pipefail

# NOTE: avoid collisions with any of the other server_ scripts
EC_VALIDATION=3

die() { echo FATAL "$@" >&2; exit 1; }

# sanity check the configuration dir env vars
for var_name in CA_DIR VPN_DIR; do
    if ! var_value="${!var_name}" \
    || [ -z "$var_value" ] \
    || [ "$var_value" != "$( echo "$var_value" | tr -d '\r\n' )" ] \
    || [ ! -d "${var_value}" ] ; then
        die "unexpected env ${var_name}=${var_value}"
    fi
done

# other scripts assume that this list doesn't change between executions of this script (as in the order and the number
# of files), since it is used to perform update logic
files=(
    "$CA_DIR/ca.key"
    "$CA_DIR/ca.csr"
    "$CA_DIR/ca.crt"
    "$CA_DIR/ca.srl"
    "$CA_DIR/ca.crl"
    "$CA_DIR/index.txt"
    "$CA_DIR/index.txt.attr"
    "$CA_DIR/certs.tar.gz"
    "$VPN_DIR/dh2048.pem"
    "$VPN_DIR/server.key"
    "$VPN_DIR/server.csr"
    "$VPN_DIR/server.crt"
) \
&& printf '%s\n' "${files[@]}" \
|| die "failed to list server config files"

# --- (PRE)NORMALISATION ---

# archive "$CA_DIR/certs" -> "$CA_DIR/certs.tar.gz"
# migration case: avoid failing if missing both
if [ ! -f "$CA_DIR/certs.tar.gz" ]; then
    mkdir -p -- "$CA_DIR/certs" \
    || die "failed to ensure directory: $CA_DIR/certs"
fi
if [ -d "$CA_DIR/certs" ]; then
    # should not run at all when loading from the storage api, aside from the migration case
    chmod -R 644 -- "$CA_DIR/certs" \
    && chmod 755 -- "$CA_DIR/certs" \
    && tar -C "$CA_DIR" -chzf "$CA_DIR/certs.tar.gz" certs \
    || die "failed to create certs archive: $CA_DIR/certs.tar.gz"
fi

# remove invalid files: various commands (openssl etc) seem to truncate / create 0 byte files on failure
for f in \
    "$CA_DIR/ca."{key,csr,crt,srl,crl} \
    "$VPN_DIR/dh2048.pem" \
    "$VPN_DIR/server."{key,csr,crt}
do
    if [ -f "$f" ] && ! { f_size="$( wc --bytes < "$f" )" && [ "$f_size" -gt 0 ]; }; then
        rm -rf "$f" \
        || die "failed to remove empty file: $f"
    fi
done

# the 2.0.1 release did not configure this file as we want it to, so patch it here
# this thing ain't really documented but apparently this is the only option that it allows configuration of anyway
# see https://www.openssl.org/docs/man1.1.0/man1/ca.html
{ [ "$( cat "$CA_DIR/index.txt.attr" 2>/dev/null )" = "unique_subject = no" ] || tee "$CA_DIR/index.txt.attr" >/dev/null <<<"unique_subject = no"; } \
|| die "failed to ensure index.txt.attr"

# --- VALIDATION ---

ec=0
for f in "${files[@]}"; do
    if [ ! -f "$f" ]; then
        ec="$EC_VALIDATION"
        echo "WARNING server validation error: missing file $f" >&2
    fi
done
if [ "$ec" -ne 0 ]; then
    exit "$ec"
fi

# --- (POST)NORMALISATION ---

# all files should exist at this point
chown root: "${files[@]}" \
&& chmod 400 \
    "$CA_DIR/ca.key" \
    "$VPN_DIR/server.key" \
&& chmod 444 \
    "$CA_DIR/ca.csr" \
    "$CA_DIR/ca.crt" \
    "$CA_DIR/ca.srl" \
    "$CA_DIR/ca.crl" \
    "$CA_DIR/index.txt" \
    "$CA_DIR/index.txt.attr" \
    "$VPN_DIR/dh2048.pem" \
    "$VPN_DIR/server.csr" \
    "$VPN_DIR/server.crt" \
|| die "failed to ensure permissions"
